const models = require('../models/estudantes')
const bcrypt = require('bcryptjs');
const yup = require('yup')
const logger = require('../services/loggerService'); 
const response = require("../constants/response");
const sendRequestOnMicroservices = require("../helpers/sendRequestOnMicroservices"); 
const { parse, isDate } = require("date-fns");
const StrengthSchecker = require('../helpers/StrengthSchecker');

function parseDateString(value, originalValue) {
  const parsedDate = isDate(new Date(originalValue))
    ? new Date(originalValue)
    : parse(originalValue, "yyyy-MM-dd", new Date());

  return parsedDate;
}

module.exports.getEstudantes = async function(req, res, next) { 
  try {  
      logger("SERVIDOR:").info(`Buscar os afiliados`)
      const {pagina, limite, total_registros, nome = '', nif = '', email = '', telefone = '', data_nascimento = '', genero = '', status = '', criacao = ''} = req.query
      const results = await models.getEstudantes(pagina, limite, total_registros, nome, nif, email, telefone, data_nascimento, genero, status, criacao)
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliados").error(`Erro buscar afiliados ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getEstudantesId = async function(req, res, next) {
  try {  

      logger("SERVIDOR:").info(`Buscar os afiliados`)
      const {id_estudantes} = req.params
      const results = await models.getEstudantesId(id_estudantes)
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliados").error(`Erro buscar afiliados ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}


module.exports.postEstudantes = async function(req, res, next) {
  try {

      logger("SERVIDOR:").info(`Buscar os afiliados`)
      const dados = req.body
      const phoneRegExp = /^((\\+[1-9]{1,4}[ \\-]*)|(\\([0-9]{2,3}\\)[ \\-]*)|([0-9]{2,4})[ \\-]*)*?[0-9]{3,4}?[ \\-]*[0-9]{3,4}?$/
      
      const schemaEstudantes = yup.object().shape({
        nome: yup.string().required(),
        telefone: yup.string().matches(phoneRegExp, 'Phone number is not valid').required(),
        data_nascimento: yup.date().transform(parseDateString).required(),
        nif: yup.string().required(),
        status: yup.mixed().oneOf(['ativo', 'inativo']).default('ativo'), 
        foto_perfil: yup.string(),
        senha: yup.string().required(),
        confirmar_senha: yup.string().oneOf([yup.ref("senha")]).required(),
        genero: yup.mixed().oneOf(['masculino','feminino','outro','nao_informado']).default('ativo'),
        email: yup.string().email().required(),
      })

      logger("SERVIDOR:postAfiliados").debug(`Á validar os dados ${JSON.stringify(dados)}`)
      const validar = await schemaEstudantes.validate(dados)

      logger("SERVIDOR:postAfiliados").debug(`Fortificando a senha`)
      const passCheck = await StrengthSchecker(validar.senha)

      if(passCheck.bg === "error"){

        logger("SERVIDOR:postAfiliados").info(`Senha para o cliente é muito fraca`)         
        const rs = response("erro", 406, "Senha para o cliente é muito fraca");
        res.status(rs.statusCode).json(rs)         

        return
      }

      var salt = bcrypt.genSaltSync(10);
      var hash = bcrypt.hashSync(validar.senha, salt);
      
      delete validar.confirmar_senha
      delete validar.senha
      
      const result = await models.postEstudantes({...validar, senha:hash}, req)

      var wk = result.webhook
      var lg = result.logs
      var nt = result.notification
      
      delete result.webhook
      delete result.logs
      delete result.notification
      
      res.status(result.statusCode).json(result)          
      if(result.status == "sucesso"){
        sendRequestOnMicroservices({lg, nt, wk})
      }

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:postClientes").error(`Erro ao cadastrar o cliente ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
  }
    
}


module.exports.patchEstudantes = async function(req, res, next) {
    try {  

        logger("SERVIDOR:").info(`Buscar os afiliados`)
        const {id_estudantes} = req.params
        const dados = req.body
        const phoneRegExp = /^((\\+[1-9]{1,4}[ \\-]*)|(\\([0-9]{2,3}\\)[ \\-]*)|([0-9]{2,4})[ \\-]*)*?[0-9]{3,4}?[ \\-]*[0-9]{3,4}?$/
        
        const schemaEstudantes = yup.object().shape({
          nome: yup.string(),
          telefone: yup.string().matches(phoneRegExp, 'Phone number is not valid'),
          data_nascimento: yup.date().transform(parseDateString),
          nif: yup.string(),
          status: yup.mixed().oneOf(['ativo', 'inativo']).default('ativo'), 
          foto_perfil: yup.string(),
          senha: yup.string(),
          confirmar_senha: yup.string().oneOf([yup.ref("senha")]),
          genero: yup.mixed().oneOf(['masculino','feminino','outro','nao_informado']).default('ativo'),
          email: yup.string().email(),
        })

        logger("SERVIDOR:postAfiliados").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaEstudantes.validate(dados)

        if(Object.keys(validar).includes('senha')){
        
          logger("SERVIDOR:patchClientes").debug(`Fortificando a senha`)
            const passCheck = await StrengthSchecker(validar?.senha)
            
            if(passCheck.bg === "error"){

              logger("SERVIDOR:patchClientes").info(`Senha para o cliente é muito fraca`)         
              const rs = response("erro", 406, "Senha para o cliente é muito fraca");
              res.status(rs.statusCode).json(rs)         
    
              return
            }

            var salt = bcrypt.genSaltSync(10);
            var hash = bcrypt.hashSync(validar.senha ? validar.senha : "1234", salt);
            validar.senha = hash
        }

        delete validar.confirmar_senha
        
        const result = await models.patchEstudantes(id_estudantes, validar, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)          
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk})
        }

    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:postClientes").error(`Erro ao cadastrar o cliente ${error.message}`)
  
        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
    }
    
  }

module.exports.deleteEstudantes = async function(req, res, next) {
    try {  

        logger("SERVIDOR:").info(`Buscar os afiliados`)
        const {id_estudantes} = req.params
        const result = await models.deleteEstudantes(id_estudantes, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)          
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk})
        }

    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:getAfiliados").error(`Erro buscar afiliados ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}