const models = require('../models/empresas')
const response = require("../constants/response");
const bcrypt = require('bcryptjs');
const path = require('path');
const yup = require('yup')
const {client} = require('../utils/rabbitMQ');
const logger = require('../services/loggerService'); 
const sendRequestOnMicroservices = require("../helpers/sendRequestOnMicroservices"); 
const StrengthSchecker = require('../helpers/StrengthSchecker');


module.exports.getEmpresas = async function(req, res, next) {
  try{
      logger("SERVIDOR:Empresas").info("Buscar clientes")
      const {pagina, limite, nomeEmpresa = '', nif = '', email = '', entidade = '', responsavel = ''} = req.query
      const results = await models.getEmpresas(pagina, limite, nomeEmpresa, nif, email, entidade, responsavel); 
      res.status(results.statusCode).json(results)
    
  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:Empresas").error(`Erro ao buscar clientes ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getEmpresasId = async function(req, res, next) {
  try{

    logger("SERVIDOR:EmpresasId").info("Buscar cliente pelo Id")
    const {empresaId} = req.params

    const results = await models.getEmpresasID(empresaId);
    res.status(results.statusCode).json(results)
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:EmpresasId").error(`Erro ao buscar cliente pelo Id ${error.message}`)
  }
    
}

module.exports.getEmpresasEntidade = async function(req, res, next) {
  try{
    const {empresaId} = req.params
    const results = await models.getEmpresasEntidade(empresaId)
    res.status(results.statusCode).json(results)
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getEmpresasHash = async function(req, res, next) {
  try{
    logger("SERVIDOR:EmpresasHash").info(`Buscar dados do hash do cliente`)
    const {hash} = req.params
    const results = await models.getEmpresasHash(hash);
    res.status(results.statusCode).json(results)
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:EmpresasHash").error(`Erro ao buscar clientes ${error.message}`)
  }
    
}

module.exports.getEmpresasEmail = async function(req, res, next) {
  try{
    logger("SERVIDOR:EmpresasEmail").info(`Buscar dados do percentual de uso`)
    const {email, canal} = req.params
    const results = await models.getEmpresasEmail(email, canal);
    res.status(results.statusCode).json(results)
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:EmpresasEmail").error(`Erro ao buscar percentual de uso ${error.message}`)
  }
    
}

module.exports.recuperarSenha = async function(req, res, next) {
  try{

    logger("SERVIDOR:recuperarSenha").info(`Realizando a recuperação de senhaEmpresa`)
    const {emailEmpresa, canal} = req.body

    const schemaRecuperarSenha = yup.object().shape({
      emailEmpresa: yup.string().required(),
      canal: yup.mixed().oneOf(['Whatsapp', 'E-mail', 'SMS']).required(), 
    })

    logger("SERVIDOR:recuperarSenha").debug(`Á validar os dados ${JSON.stringify(req.body)}`)
    const validar = await schemaRecuperarSenha.validate(req.body)
    const result = await models.recuperarSenha(validar.emailEmpresa, validar.canal, req)
    
    var wk = result.webhook
    var lg = result.logs
    var nt = result.notification
    
    delete result.webhook
    delete result.logs
    delete result.notification
    
    res.status(result.statusCode).json(result)
    if(result.status == "sucesso"){
      sendRequestOnMicroservices({lg, nt, wk})
    }
     
  } catch (error) {
    console.error(error.message)
    logger("SERVIDOR:recuperarSenha").error(`Erro ao recuperar a senhaEmpresa ${error.message}`)

    if(error?.path){
      const rs = response("erro", 412, error.message);
      res.status(rs.statusCode).json(rs)        
    }else{  
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
    }
  }
    
}

module.exports.redifinirSenha = async function(req, res, next) {
  try{

    logger("SERVIDOR:redifinirSenha").info(`Realizando a redifinição de senhaEmpresa`)
    const dados = req.body

    const schemaRedifinirSenha = yup.object().shape({
      codigo_seguranca: yup.string().min(6).required(),
      entidade: yup.string().min(1).required(), 
    })

    logger("SERVIDOR:redifinirSenha").debug(`Á validar os dados ${JSON.stringify(dados)}`)
    const validar = await schemaRedifinirSenha.validate(dados)
    const result = await models.redifinirSenha(validar.codigo_seguranca, validar.entidade, req);

    var wk = result.webhook
    var lg = result.logs
    var nt = result.notification
    
    delete result.webhook
    delete result.logs
    delete result.notification
    
    res.status(result.statusCode).json(result)
    if(result.status == "sucesso"){
      sendRequestOnMicroservices({lg, nt, wk})
    }
    
  } catch (error) {
    console.error(error.message)
    logger("SERVIDOR:redifinirSenha").error(`Erro ao redifinir a senhaEmpresa ${error.message}`)

    if(error?.path){
      const rs = response("erro", 412, error.message);
      res.status(rs.statusCode).json(rs)        
    }else{  
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
    }
  }
    
}

module.exports.postEmpresas = async function(req, res, next) { 
    
   try {

      logger("SERVIDOR:postEmpresas").info(`Iniciando cadastrado o cliente`)

      const dados =  req.body
      const dadosHeader =  req.headers
      const phoneRegExp = /^((\\+[1-9]{1,4}[ \\-]*)|(\\([0-9]{2,3}\\)[ \\-]*)|([0-9]{2,4})[ \\-]*)*?[0-9]{3,4}?[ \\-]*[0-9]{3,4}?$/

      const schemaEntidades = yup.object().shape({
        emailEmpresa: yup.string().email().required(),
        senhaEmpresa: yup.string().required(),
        confirmar_senha: yup.string().oneOf([yup.ref("senhaEmpresa"), "As senhas não batem"]).required(),
        nomeEmpresa: yup.string().min(5).required(),
        nifEmpresa: yup.string().required(),
        telefoneEmpresa: yup.string().matches(phoneRegExp, 'Número de telefone invalido').required(),
        telefoneEmpresaAlt: yup.string().matches(phoneRegExp, 'Número de telefone invalido'),
        empresaId: yup.number().required(),
        enderecoEmpresa: yup.string().min(5).required(),
        IBAN: yup.string(),
        referenciaEMIS: yup.string()
      })
      

      const schemaEntidadesHeader = yup.object().shape({
        criador: yup.number().required()
      })

      logger("SERVIDOR:postEmpresas").debug(`Á validar os dados ${JSON.stringify(dados)}`)
      const validar = await schemaEntidades.validate(dados)
      const validarHeader = await schemaEntidadesHeader.validate(dadosHeader)

      logger("SERVIDOR:postEmpresas").debug(`Fortificando a senhaEmpresa`)
      const passCheck = await StrengthSchecker(validar.senhaEmpresa)

      if(passCheck.bg === "error"){

        logger("SERVIDOR:postEmpresas").info(`Senha para o cliente é muito fraca`)         
        const rs = response("erro", 406, "Senha para o cliente é muito fraca");
        res.status(rs.statusCode).json(rs)         

        return
      } 

      var salt = bcrypt.genSaltSync(10);
      var hash = bcrypt.hashSync(validar.senhaEmpresa, salt);

      if(validar?.nomeEmpresa){
        validar.nomeEmpresa = validar.nomeEmpresa.toUpperCase().trim();
      }
      
      delete validar.confirmar_senha
      const result = await models.postEmpresas({...validar, nomeEmpresa: validar.nomeEmpresa.toUpperCase().trim(), senhaEmpresa:hash, criado_por: validarHeader.criador}, req)  
      
      var wk = result.webhook
      var lg = result.logs
      var nt = result.notification  
      
      delete result.webhook
      delete result.logs
      delete result.notification

      res.status(result.statusCode).json(result)

      if(result.status == "sucesso"){

        sendRequestOnMicroservices({lg, nt, wk})
      
        
      }    
      
      
   } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:postEmpresas").error(`Erro ao cadastrar o cliente ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
   }
    
}

module.exports.loginEmpresas = async function(req, res, next) {
    
   try {

      logger("SERVIDOR:loginEmpresas").info(`Realizando o login`)
      const dados =  req.body      
      
      const schemaLogin = yup.object().shape({
        emailEmpresa: yup.string().required(),
        senhaEmpresa: yup.string().required(),
      })

      logger("SERVIDOR:loginEmpresas").debug(`Á validar os dados ${JSON.stringify(dados)}`)
      const validar = await schemaLogin.validate(dados)
      const result = await models.loginEmpresas(validar, req) 
      
      var wk = result.webhook
      var lg = result.logs
      var nt = result.notification
      
      delete result.webhook
      delete result.logs
      delete result.notification
      
      res.status(result.statusCode).json(result)
      if(result.status == "sucesso"){
        sendRequestOnMicroservices({lg, nt, wk})
      }
      
   } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:loginEmpresas").error(`Erro ao realizar o login ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
   }
    
}

module.exports.logoutEmpresas = async function(req, res, next) {
    
   try {

      logger("SERVIDOR:logoutEmpresas").info("Realizando o login")
      const dados = req.body      

      const schemaLogout = yup.object().shape({
        entidade: yup.string().required(),
      })

      logger("SERVIDOR:logoutEmpresas").debug(`Á validar os dados ${JSON.stringify(dados)}`)
      const validar = await schemaLogout.validate(dados)
      const result = await models.logoutEmpresas(validar.entidade, req) 
      
      var wk = result.webhook
      var lg = result.logs
      var nt = result.notification
      
      delete result.webhook
      delete result.logs
      delete result.notification
      
      res.status(result.statusCode).json(result)
      if(result.status == "sucesso"){
        sendRequestOnMicroservices({lg, nt, wk})
      }
      

        return result
      
   } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:logoutEmpresas").error(`Erro ao realizar o logout ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
   }
    
}

module.exports.comunicarEmail = async function(req, res, next) {
    
   try {
        logger("SERVIDOR:comunicarEmail").info(`Realizando o login`)
        const dados = req.body 

        const schemaEmail = yup.object().shape({
          email: yup.array(yup.string().email().required()).required(),
          assunto: yup.string().required(),
          html: yup.string().required()
        })

        logger("SERVIDOR:comunicarEmail").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaEmail.validate(dados)
        const result = await models.comunicarEmail(validar, req) 
        
        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk})
        }
      
   } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:comunicarEmail").error(`Erro ao realizar o email ${error.message}`)

        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
   }
    
}

module.exports.activarPorCodigo = async function(req, res, next) {
    
   try {
        logger("SERVIDOR:activarPorCodigo").info(`Realizando o login`)
        const dados = req.body 

        const schemaEmail = yup.object().shape({
          codigo_confirmacao: yup.string().required()
        })

        logger("SERVIDOR:activarPorCodigo").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaEmail.validate(dados)
        const result = await models.activarPorCodigo(validar, req) 
        
        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk})
        }
      
   } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:activarPorCodigo").error(`Erro ao realizar o email ${error.message}`)

        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
   }
    
}

module.exports.patchEmpresas = async function(req, res, next) { 
      try {

        logger("SERVIDOR:patchEmpresas").info(`Iniciando actualização do cliente`)
        const {empresaId} = req.params
        const dados = req.body
        const phoneRegExp = /^((\\+[1-9]{1,4}[ \\-]*)|(\\([0-9]{2,3}\\)[ \\-]*)|([0-9]{2,4})[ \\-]*)*?[0-9]{3,4}?[ \\-]*[0-9]{3,4}?$/

        const schemaEntidades = yup.object().shape({
          emailEmpresa: yup.string().email(),
          senhaEmpresa: yup.string(),
          confirmar_senha: yup.string().oneOf([yup.ref("senhaEmpresa")]),
          nomeEmpresa: yup.string().min(5),
          nifEmpresa: yup.string(),
          telefoneEmpresa: yup.string().matches(phoneRegExp, 'Phone number is not valid'),
          telefoneEmpresaAlt: yup.string().matches(phoneRegExp, 'Phone number is not valid'),
          empresaId: yup.number(),
          tipoEmpresa: yup.mixed().oneOf(['Particular','Empresa']).default('Empresa'), 
          enderecoEmpresa: yup.string().min(5),          
          cidadeEmpresa: yup.string().min(5),
          provincia: yup.string(),
          paisEmpresa: yup.string(),
          IBAN: yup.string(),
          referenciaEMIS: yup.string()
        })

        logger("SERVIDOR:patchEmpresas").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaEntidades.validate(dados)
        
        if(Object.keys(validar).includes('senhaEmpresa')){          

          logger("SERVIDOR:patchEmpresas").debug(`Fortificando a senhaEmpresa`)
          const passCheck = await StrengthSchecker(validar?.senhaEmpresa)
          
          if(passCheck.bg === "error"){

            logger("SERVIDOR:patchEmpresas").info(`Senha para o cliente é muito fraca`)         
            const rs = response("erro", 406, "Senha para o cliente é muito fraca");
            res.status(rs.statusCode).json(rs)         
  
            return
          }

          var salt = bcrypt.genSaltSync(10);
          var hash = bcrypt.hashSync(validar.senhaEmpresa, salt);
          validar.senhaEmpresa = hash

        }
        
        delete validar.confirmar_senha

        if(validar?.nomeEmpresa){
          validar.nomeEmpresa = validar.nomeEmpresa.toUpperCase().trim();
        }

        const result = await models.patchEmpresas(empresaId, validar, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)

        if(result.status == "sucesso"){

          if(dados?.gpo_numero_comerciante){

            if(result.entidadeGPO){
              const queue = 'reference-entities';   
              const channel = await client.rabbitMQ.createChannel() 
              await channel.assertQueue(queue)
              const payload = JSON.stringify({
                kind: "entities:updated",
                body:{
                  "storeName": dados.nomeEmpresa,
                  "merchantCode":dados.gpo_numero_comerciante,
                  "cardNumber":dados.gpo_numero_cartao,
                  "bankNumber":dados.gpo_numero_banco,
                  "email": dados.email,
                  "numbersOfPos":dados.gpo_numero_POS,
                  "numberOfEstablishments": dados.gpo_numero_establecimento
                }
              })
                channel.sendToQueue(queue, Buffer.from(payload));
                console.log("Actualizar Empresas: Enviado para fila")
              }
          }

          if(dados?.gpo_numero_comerciante){

              if(!result.entidadeGPO){
                const queue = 'reference-entities';      
                const channel = await client.rabbitMQ.createChannel()
                await channel.assertQueue(queue)
                const payload = JSON.stringify({
                  kind: "entities:created",
                  body:{
                    "storeName": dados.nomeEmpresa,
                    "merchantCode":dados.gpo_numero_comerciante,
                    "cardNumber":dados.gpo_numero_cartao,
                    "bankNumber":dados.gpo_numero_banco,
                    "email": dados.email,
                    "numbersOfPos":dados.gpo_numero_POS,
                    "numberOfEstablishments": dados.gpo_numero_establecimento
                  }
                })
                  channel.sendToQueue(queue, Buffer.from(payload));
                  console.log("Cadastrar Empresas: Enviado para fila")
              }
          }
          
          sendRequestOnMicroservices({lg, nt, wk})

        }
        

      } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchEmpresas").error(`Erro ao actualizar o cliente ${error.message}`)

        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
      }
    
}

module.exports.patchEmpresasRedifinirSenha = async function(req, res, next) { 
      try {

        logger("SERVIDOR:patchEmpresasRedifinirSenha").info(`Iniciando actualização de senhaEmpresa`)
        const {entidade} = req.params
        const dados = req.body

        const schemaEntidades = yup.object().shape({
          senhaEmpresa: yup.string().required(),
          confirmar_senha: yup.string().oneOf([yup.ref("senhaEmpresa")], "As senhas não conscidem").required()
        })

        logger("SERVIDOR:patchEmpresasRedifinirSenha").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaEntidades.validate(dados)
        
        if(Object.keys(dados).includes('senhaEmpresa')){          

          logger("SERVIDOR:patchEmpresasRedifinirSenha").debug(`Fortificando a senhaEmpresa`)
          const passCheck = await StrengthSchecker(validar?.senhaEmpresa)
          
          if(passCheck.bg === "error"){

            logger("SERVIDOR:patchEmpresasRedifinirSenha").info(`Senha para o cliente é muito fraca`)         
            const rs = response("erro", 406, "Senha para o cliente é muito fraca");
            res.status(rs.statusCode).json(rs)         
  
            return
          }

          var salt = bcrypt.genSaltSync(10);
          var hash = bcrypt.hashSync(validar.senhaEmpresa, salt);
          dados.senhaEmpresa = hash

        }
        
        delete validar.confirmar_senha
        const result = await models.patchEmpresasRedifinirSenha(entidade, validar, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)

        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk})
        }
        

      } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchEmpresasRedifinirSenha").error(`Erro ao actualizar da senhaEmpresa ${error.message}`)

        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
      }
    
}

module.exports.patchEmpresasTrocarSenhaPadrao = async function(req, res, next) { 
      try {

        logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").info(`Iniciando actualização da senhaEmpresa padrão`)
        const {entidade} = req.params
        const dados = req.body

        const schemaEntidades = yup.object().shape({
          senhaEmpresa: yup.string().required(),
          confirmar_senha: yup.string().oneOf([yup.ref("senhaEmpresa")]).required(),
        })

        logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaEntidades.validate(dados)                 

        logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").debug(`Fortificando a senhaEmpresa`)
        const passCheck = await StrengthSchecker(validar?.senhaEmpresa)
        
        if(passCheck.bg === "error"){

          logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").info(`Senha para o cliente é muito fraca`)         
          const rs = response("erro", 406, "Senha para o cliente é muito fraca");
          res.status(rs.statusCode).json(rs)         

          return
        }

        var salt = bcrypt.genSaltSync(10);
        var hash = bcrypt.hashSync(validar.senhaEmpresa, salt);
        dados.senhaEmpresa = hash
        
        delete validar.confirmar_senha
        const result = await models.patchEmpresasTrocarSenhaPadrao(entidade, validar, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)

        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk})
        }
        

      } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").error(`Erro ao actualizar da senhaEmpresa padrão ${error.message}`)

        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
      }
    
}

module.exports.patchEmpresasVerificarSenhaActual = async function(req, res, next) { 
      try {

        logger("SERVIDOR:patchEmpresasVerificarSenhaActual").info(`Iniciando actualização da senhaEmpresa padrão`)
        const {entidade} = req.params
        const dados = req.body

        const schemaEntidades = yup.object().shape({
          senha_actual: yup.string().required(),
        })

        logger("SERVIDOR:patchEmpresasVerificarSenhaActual").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaEntidades.validate(dados)
        
        const result = await models.patchEmpresasVerificarSenhaActual(entidade, validar.senha_actual, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)

        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk})
        }
        

      } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchEmpresasVerificarSenhaActual").error(`Erro ao verificar da senhaEmpresa actual ${error.message}`)

        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
      }
    
}

module.exports.patchEmpresasAlterarSenha = async function(req, res, next) { 
      try {

        logger("SERVIDOR:patchEmpresasAlterarSenha").info(`Iniciando actualização da senha padrão`)
        const {entidade} = req.params
        const dados = req.body

        const schemaEntidades = yup.object().shape({
          senha_actual: yup.string().required(),
          senha: yup.string().required(),
          confirmar_senha: yup.string().oneOf([yup.ref("senha")], "As senhas não conscidem").required()
        })

        logger("SERVIDOR:patchEmpresasRedifinirSenha").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaEntidades.validate(dados)
        
        if(Object.keys(dados).includes('senha')){          

          logger("SERVIDOR:patchEmpresasRedifinirSenha").debug(`Fortificando a senha`)
          const passCheck = await StrengthSchecker(validar?.senha)
          
          if(passCheck.bg === "error"){

            logger("SERVIDOR:patchEmpresasRedifinirSenha").info(`Senha para o cliente é muito fraca`)         
            const rs = response("erro", 406, "Senha para o cliente é muito fraca");
            res.status(rs.statusCode).json(rs)         
  
            return
          }

          var salt = bcrypt.genSaltSync(10);
          var hash = bcrypt.hashSync(validar.senha, salt);
          dados.senha = hash

        }
        
        delete validar.confirmar_senha        
        const result = await models.patchEmpresasAlterarSenha(entidade, validar.senha_actual, validar.senha, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)

        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk})
        }
        

      } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchEmpresasAlterarSenha").error(`Erro ao verificar da senha actual ${error.message}`)

        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
      }
    
}

exports.mudarFotoEmpresas = async function (req, res){
  try {
      
      logger("SERVIDOR:mudarFotoEmpresas").info(`Iniciando a mudança de imagem`)
      let foto_perfil = '';
      const {entidade} = req.params 
      
      if (req.files && req.files.foto_perfil) {
  
          let targetFile = req.files.foto_perfil
          const ext =  path.extname(targetFile.name)
          const filename = targetFile.md5+ext
          foto_perfil =  filename;  

          
          const extensoesPermitidas = ['.png','.jpg', '.jpeg', '.svg','.gif','.webp']
          
          if(!extensoesPermitidas.includes(ext.toLowerCase())) {
            logger("SERVIDOR:mudarFotoEmpresas").info("Seu arquivo não é permitido. Tente um desses: '.png','.jpg', '.jpeg', '.svg','.gif','.webp'")
            res.status(406).json({status:'erro',mensagem:"Seu arquivo não é permitido. Tente um desses: '.png','.jpg', '.jpeg', '.svg','.gif','.webp'"})
            return;
          }
          
          logger("SERVIDOR:mudarFotoEmpresas").debug(`Movendo o arquivo da pasta tmp do sistema para o especifico`)
          targetFile.mv(path.join(__dirname, '../assets/imgs',filename), (err) => {
              
            if (err) {
                logger("SERVIDOR:mudarFotoEmpresas").info(`Algo aconteceu de errado. Não conseguimos carregar o arquivo`)
                res.status(400).json({status:'erro',mensagem:"Algo aconteceu de errado. Não conseguimos carregar o arquivo"})
                return;
            }
              
          });
          
          logger("SERVIDOR:mudarFotoEmpresas").info(`Upload feito, Á gravar na base de dados`)
        
          const result = await models.patchEmpresasFoto(entidade, {logo: foto_perfil}, req)

          var wk = result.webhook
          var lg = result.logs
          var nt = result.notification
          
          delete result.webhook
          delete result.logs
          delete result.notification
          
          res.status(result.statusCode).json(result)
          if(result.status == "sucesso"){          
            sendRequestOnMicroservices({lg, nt, wk})
          }

      }else{
            logger("SERVIDOR:mudarFotoEmpresas").info(`Algo aconteceu de errado. Tente outra vez! Selecionou sua imagem?`)
            const rs = response("erro", 406, `Algo aconteceu de errado. Tente outra vez! Selecionou sua imagem?`);
            res.status(rs.statusCode).json(rs)
      }
              
      
  } catch (error) {
      console.log(error)
      logger("SERVIDOR:mudarFotoEmpresas").error(`Erro ao mudar a foto do cliente ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
  }
}

exports.mudarArquivoContratoEmpresas = async function (req, res){
try {
    
    logger("SERVIDOR:mudarArquivoContratoEmpresas").info(`Iniciando a mudança do pdf do contrato`)
    let arquivo_contrato = '';
    const {entidade} = req.params 
    
    if (req.files && req.files.arquivo_contrato) {

        let targetFile = req.files.arquivo_contrato
        const ext =  path.extname(targetFile.name)
        const filename = targetFile.md5+ext
        arquivo_contrato =  filename;  

        const extensoesPermitidas = ['.pdf']

        if(!extensoesPermitidas.includes(ext.toLowerCase())) {
          logger("SERVIDOR:mudarArquivoContratoEmpresas").info("Seu arquivo não é permitido. Tente um desses: '.pdf'")
          res.status(406).json({status:'erro',mensagem:"Seu arquivo não é permitido. Tente um desses: '.pdf'"})
          return;
        }
        
        logger("SERVIDOR:mudarArquivoContratoEmpresas").debug(`Movendo o arquivo da pasta tmp do sistema para o especifico`)
        targetFile.mv(path.join(__dirname, '../assets/contratos',filename), (err) => {
            
          if (err) {
              logger("SERVIDOR:mudarArquivoContratoEmpresas").info(`Algo aconteceu de errado. Não conseguimos carregar o arquivo`)
              const rs = response("erro", 400, `Algo aconteceu de errado. Não conseguimos carregar o arquivo`);
              res.status(rs.statusCode).json(rs)
              return;
          }
            
        });
        
        logger("SERVIDOR:mudarArquivoContratoEmpresas").info(`Upload feito, Á gravar na base de dados`)
      
        const result = await models.patchEmpresasArquivoContrato(entidade, {arquivo_do_contrato: arquivo_contrato}, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){          
          sendRequestOnMicroservices({lg, nt, wk})
        }
        
        // 995302104
        
    }else{
          logger("SERVIDOR:mudarArquivoContratoEmpresas").info(`Algo aconteceu de errado. Tente outra vez! Selecionou seu PDF?`)
          const rs = response("erro", 406, `Algo aconteceu de errado. Tente outra vez! Selecionou seu PDF?`);
          res.status(rs.statusCode).json(rs)
    }
        
        
} catch (error) {
    console.log(error)
    logger("SERVIDOR:patchEmpresasArquivoContrato").error(`Erro ao mudar o arquivo de contrato do cliente ${error.message}`)

    if(error?.path){
      const rs = response("erro", 412, error.message);
      res.status(rs.statusCode).json(rs)        
    }else{  
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
    }
}
}

module.exports.patchEmpresasBloquear = async function(req, res, next) {

      try {

        logger("SERVIDOR:patchEmpresasBloquear").info(`Iniciando o bloqueio de cliente`)
        const {empresaId} = req.params
        const result = await models.patchEmpresasBloquear(empresaId, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){          
          sendRequestOnMicroservices({lg, nt, wk})
        }

      } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchEmpresasBloquear").error(`Erro ao bloquear o cliente ${error.message}`)
      }
    
}

module.exports.patchEmpresasDesbloquear = async function(req, res, next) {  
      try {

        logger("SERVIDOR:patchEmpresasDesbloquear").info(`Iniciando o desbloqueio de cliente`)
        const {empresaId} = req.params
        const result = await models.patchEmpresasDesbloquear(empresaId, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){          
          sendRequestOnMicroservices({lg, nt, wk})
        }
        

      } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchEmpresasDesbloquear").error(`Erro ao desbloquear o cliente ${error.message}`)
      }
    
}

module.exports.configurarReporEmpresas = async function(req, res, next) {  
      try {

          logger("SERVIDOR:configurarReporEmpresas").info(`Iniciando a reposição do cliente`)
          const {empresaId} = req.params
          const dados = req.body

          const schemaReporEmpresas = yup.object().shape({
            tentativas_login: yup.number().max(5)
          })

          logger("SERVIDOR:configurarReporEmpresas").debug(`Á validar os dados ${JSON.stringify(dados)}`)
          const validar = await schemaReporEmpresas.validate(dados)

          const result = await models.configurarReporEmpresas(empresaId, validar.tentativas_login, req)

          var wk = result.webhook
          var lg = result.logs
          var nt = result.notification
          
          delete result.webhook
          delete result.logs
          delete result.notification
          
          res.status(result.statusCode).json(result)
          if(result.status == "sucesso"){          
            sendRequestOnMicroservices({lg, nt, wk})
          }

      } catch (error) {
          console.error(error.message)
          logger("SERVIDOR:configurarReporEmpresas").error(`Erro ao repor o cliente ${error.message}`)

          if(error?.path){
            const rs = response("erro", 412, error.message);
            res.status(rs.statusCode).json(rs)        
          }else{  
            const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
            res.status(rs.statusCode).json(rs)
          }
      }
    
}

module.exports.deleteEmpresas = async function(req, res, next) {
  try {

      logger("SERVIDOR:deleteEmpresas").info(`Iniciando a exlusão do cliente`)
      const {empresaId} = req.params
      const result = await models.deleteEmpresas(empresaId, req)

      var wk = result.webhook
      var lg = result.logs
      var nt = result.notification
      
      delete result.webhook
      delete result.logs
      delete result.notification
      
      res.status(result.statusCode).json(result)
      if(result.status == "sucesso"){

        if(result?.gpo_comerciante_hash){

            const queue = 'reference-entities';   
            const channel = await client.rabbitMQ.createChannel() 
            await channel.assertQueue(queue)
            const payload = JSON.stringify({
              kind: "entities:deleted",
              body:{
                "public_id": result.gpo_comerciante_hash
              }
            })
            channel.sendToQueue(queue, Buffer.from(payload));
            console.log("Deletar Empresas: Enviado para fila")
        }
      
        sendRequestOnMicroservices({lg, nt, wk})
      }
          

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:deleteEmpresas").error(`Erro ao deletar o cliente ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}