const database = require('../config/database')
const bcrypt = require('bcryptjs');
const { v4: uuidv4 } = require('uuid');
const path = require("path");
const response = require("../constants/response");
const logger = require('../services/loggerService');
const paginationRecords = require("../helpers/paginationRecords")
const { clientesTruesFilteres } = require('../helpers/filterResponseSQL');
require("dotenv").config({ path: path.resolve(path.join(__dirname,'../','.env')) });

module.exports.getClientes = async function(pagina, limite, nomeCliente, nifCliente, emailCliente, entidade, responsavel) {
  try {
      
      logger("SERVIDOR:cliente").debug("Selecionar da base de dados")

      const cliente = await database('cliente')
      .join('usuarios',"usuarios.id_usuarios","=","cliente.criado_por")
      .join('configuracoes',"configuracoes.clienteIdConf","=","cliente.clienteId")
      .whereLike("nomeCliente",`%${String(nomeCliente).toUpperCase()}%`)
      .whereLike("nifCliente",`%${nifCliente}%`)
      .whereLike("emailCliente",`%${emailCliente}%`)
      .whereLike("clienteId",`%${entidade}%`)
      // .whereLike("responsavel",`%${responsavel}%`)
      .orderBy('clienteId','DESC')

      const {registros} = paginationRecords(cliente, pagina, limite)

      logger("cliente").debug(`Buscar todos cliente no banco de dados com limite de ${registros.limite} na pagina ${registros.count} de registros`);
      const clientesLimite = await database('cliente')
      .join('usuarios',"usuarios.id_usuarios","=","cliente.criado_por")
      .join('configuracoes',"configuracoes.clienteIdConf","=","cliente.clienteId")
      .whereLike("nomeCliente",`%${String(nomeCliente).toUpperCase()}%`)
      .whereLike("nifCliente",`%${nifCliente}%`)
      .whereLike("emailCliente",`%${emailCliente}%`)
      .whereLike("clienteId",`%${entidade}%`)
      // .whereLike("responsavel",`%${responsavel}%`)
      .limit(registros.limite)
      .offset(registros.count)
      .orderBy('clienteId','DESC')

      const filtered = clientesTruesFilteres(clientesLimite)

      registros.total_apresentados = clientesLimite.length
      registros.nomeCliente = nomeCliente
      registros.nifCliente = nifCliente
      registros.emailCliente = emailCliente
      registros.entidade = entidade
      registros.responsavel = responsavel

      logger("SERVIDOR:cliente").info("Respondeu a solicitação")
      const rs = response("sucesso", 200, filtered, "json", { registros });
      return rs


  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:cliente").error(`Erro ao buscar cliente ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.getClientesID = async function(clienteId) {
  try {

      logger("SERVIDOR:ClientesId").debug("Selecionar da base de dados")
      const [cliente] = await database('cliente')
      .join('usuarios',"usuarios.id_usuarios","=","cliente.criado_por")
      .join('configuracoes',"configuracoes.clienteIdConf","=","cliente.clienteId")
      .where({clienteId})
      .orderBy('clienteId','DESC')
    
      logger("SERVIDOR:ClientesId").info("Respondeu a solicitação")
      const rs = response("sucesso", 200, cliente || {});          
      return rs

  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:ClientesId").error(`Erro ao buscar cliente por ID ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.getClientesEntidade = async function(clienteId) {
  try {

      logger("SERVIDOR:getClientesEntidade").debug("Á buscar os dados")
      const cliente = await database('cliente')
      .join('usuarios',"usuarios.id_usuarios","=","cliente.criado_por")
      .join('configuracoes',"configuracoes.clienteIdConf","=","cliente.clienteId")
      .where({clienteId})
      .orderBy('clienteId','DESC')  
    
      logger("SERVIDOR:getClientesEntidade").info("Respondeu a solicitação")
      const rs = response("sucesso", 200, cliente, "json");          
      return rs

  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:getClientesEntidade").error(`Erro ao buscar cliente por entidade ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.getClientesHash = async function(hash) {
  try {

      logger("SERVIDOR:ClientesHash").debug("Verificar se existencia do cliente")
      const result  = await database('cliente')
      .where({endereco_mac_unico:hash})
      .andWhere({bloqueio: '0'});

      if(result.length == 0 ){
        logger("SERVIDOR:ClientesHash").info("Cliente está bloqueado")
        const rs = response("erro", 403, "Cliente está bloqueado");
        return rs

      }else{

        logger("SERVIDOR:ClientesHash").debug("Buscar a configuração do cliente")
        const [verifivarPagamentoTempoReal] = await database('configuracoes').where({clienteIdConf: result[0].clienteId}) 

        console.log(verifivarPagamentoTempoReal?.servico_principal)

        if((verifivarPagamentoTempoReal?.servico_principal == "false")){
          logger("SERVIDOR:loginClientes").warn("Serviço de gestão inactivo")
          const rs = response("erro", 423, "Serviço de gestão inactivo");
          return rs     
        }

        if(verifivarPagamentoTempoReal?.tentativas_login > 0){

          logger("SERVIDOR:ClientesHash").debug("Selecionar da base de dados")
          const [cliente] = await database('cliente')
          .join('configuracoes',"configuracoes.clienteIdConf","=","cliente.clienteId")
          .join('conta_saldo',"conta_saldo.clienteIdSaldo","=","cliente.clienteId")
          .where({endereco_mac_unico:hash})
          .orderBy('clienteId','DESC')  
        
          logger("SERVIDOR:ClientesHash").info("Respondeu a solicitação")
          const rs = response("sucesso", 200, cliente, "json");          
          return rs

        }else{
          logger("SERVIDOR:ClientesHash").warn("Cliente está bloqueiado")
          const rs = response("erro", 403, "Seu perfil se encontra bloqueado. Por favor entre em contacto com administrador");
          return rs
        }

      }

  } catch (erro) { 
      console.log(erro)
      logger("SERVIDOR:ClientesHash").error(`Erro ao buscar cliente por hash ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.getClientesEmail = async function(emailCliente) { 
  try {

      logger("SERVIDOR:ClientesEmail").debug("Verificar se existencia do cliente")
      const cliente = await database('cliente')
      .join('usuarios',"usuarios.id_usuarios","=","cliente.criado_por")
      .join('configuracoes',"configuracoes.clienteIdConf","=","cliente.clienteId")
      .where({emailCliente})
      .orderBy('clienteId','DESC')
    
      logger("SERVIDOR:ClientesEmail").info("Respondeu a solicitação")
      const rs = response("sucesso", 200, cliente, "json");          
      return rs
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:ClientesEmail").error(`Erro ao buscar cliente ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
    
}

module.exports.recuperarSenha = async function(emailCliente, canal, req) {
  try {

    logger("SERVIDOR:recuperarSenha").debug(`Verificar a existencia do cliente pelo emailCliente ${emailCliente}`)
    const cliente = await database('cliente')
    .where({emailCliente})
    .orWhere({telefoneCliente: emailCliente})
    .orderBy('clienteId','DESC')
    
    logger("SERVIDOR:recuperarSenha").debug(`Gerando um codigo de 6 digitos`)
    let codigo_seguranca = String(Math.random()).replaceAll(".","").substr(0,6);
    
    var now = new Date();
    var time = now.getTime();
    var expireTime = time + 1000*60*10;
    
    let tempo_de_vida_codigo_seguranca = expireTime;
    
    logger("SERVIDOR:recuperarSenha").info(`Codigo gerado: ${codigo_seguranca} para ${tempo_de_vida_codigo_seguranca} de vida util`)
    if(cliente.length > 0) {

      logger("SERVIDOR:recuperarSenha").debug(`Verficar duplicidade de codigo`)
      const codigounico = await database('configuracoes').where({codigo_seguranca})
      
      logger("SERVIDOR:recuperarSenha").debug(`Gerar outro codigo`)
      while(codigounico.length > 0) {
          codigo_seguranca = String(Math.random()).replaceAll(".","").substr(0,6);
      }
      
      logger("SERVIDOR:recuperarSenha").debug(`Actualizando e gravando na base de dados`)
      await database('configuracoes').where({clienteIdConf: cliente[0].clienteId}).update({codigo_seguranca, tempo_de_vida_codigo_seguranca})
      
      
        if(canal === "Whatsapp") {
            logger("SERVIDOR:recuperarSenha").info(`Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`)
            const rs = response("sucesso", 202, `Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`, 'json',{
              notification: {efeito: {contacto: cliente[0].contacto, clienteId: cliente[0].clienteId, emailCliente: cliente[0].emailCliente, codigo_seguranca }, para:"codigoSeguranca", mensagem: 'null', canal:"whatsapp"},
              info: {entidade:cliente[0].clienteId, tempo_de_vida_codigo_seguranca},
              logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SENDSECURITYCODE" , tabela: "cliente", informacao: {emailCliente, tempo_de_vida_codigo_seguranca, codigo_seguranca, canal}, entidade: cliente[0].clienteId}
            });
            return rs
        }
        else if(canal === "E-mail") {
          

            logger("SERVIDOR:recuperarSenha").info(`Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`)
            const rs = response("sucesso", 202, `Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`, 'json',{
              notification: {efeito: {empresa: cliente[0].nomeCliente, email: emailCliente, codigo_seguranca }, para:"codigoSeguranca", mensagem: 'null', canal:"email"},
              info: {entidade:cliente[0].clienteId, tempo_de_vida_codigo_seguranca},
              logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SENDSECURITYCODE" , tabela: "cliente", informacao: {emailCliente, tempo_de_vida_codigo_seguranca, codigo_seguranca, canal}, entidade: cliente[0].clienteId}
            });
            return rs
        }
        else if(canal.toUpperCase() === "SMS") {
            
            logger("SERVIDOR:recuperarSenha").info(`Caro(a) cliente, realizou em pedido de alteração de sua senhaCliente. O codigo é:  ${codigo_seguranca}`)
            const mensagem = `Caro(a) cliente, realizou em pedido de alteração de sua senhaCliente. O codigo é:  ${codigo_seguranca}`;
            const rs = response("sucesso", 202, `Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`, 'json',{
              notification: {efeito: {...cliente[0]}, para:"RECUPERACAO", mensagem, canal:"sms"},
              info: {entidade:cliente[0].clienteId, tempo_de_vida_codigo_seguranca},
              logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SENDSECURITYCODE" , tabela: "cliente", informacao: {emailCliente, tempo_de_vida_codigo_seguranca, codigo_seguranca, canal}, entidade: cliente[0].clienteId}
            });
            return rs
        }
        else{
            logger("SERVIDOR:recuperarSenha").info(`Não conseguimos enviar o codigo de segurança para o seu ${canal}`)
            const rs = response("erro", 406, `Não conseguimos enviar o codigo de segurança para o seu ${canal}`);
            return rs 
        }
        
    }else{
    
      logger("SERVIDOR:recuperarSenha").info(`Não conseguimos seu E-mail nos nossos registros`)
      const rs = response("erro", 409, 'Não conseguimos seu E-mail nos nossos registros');
      return rs             
      
    }    
    
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:recuperarSenha").error(`Erro ao recuperar a senhaCliente ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs 
  }
    
}

module.exports.redifinirSenha = async function(codigo_seguranca, entidade, req) {
  try {

    logger("SERVIDOR:redifinirSenha").debug(`Verificar o codigo de segurança`)
    const codigo = await database('configuracoes').where({codigo_seguranca}).andWhere({clienteIdConf: entidade})
    
    if(codigo.length > 0) {
    
      var now = new Date();
      var time = now.getTime();
      
      if(time > codigo[0].tempo_de_vida_codigo_seguranca){
        logger("SERVIDOR:redifinirSenha").info(`Codigo de segurança expirado. Por favor repita os passos novamente`)
        const rs = response("erro", 406, 'Codigo de segurança expirado. Por favor repita os passos novamente!','json',{
          logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SECURITYCODEEXPIRE" , tabela: "cliente", informacao: {tempo_de_vida_codigo_seguranca: codigo[0].tempo_de_vida_codigo_seguranca, codigo_seguranca}, entidade: entidade}
        });
        return rs
      }
      

      logger("SERVIDOR:redifinirSenha").debug(`Actualizar o codigo de segurança para 0000`)
      await database('configuracoes').andWhere({clienteIdConf: entidade}).update({codigo_seguranca:"000000"}) 
      
      if(codigo_seguranca != "0000"){
        logger("SERVIDOR:redifinirSenha").debug(`Actualizar e resetar as ${process.env.LIMITE_TENTATIVAS_LOGIN} tentativas`)
        await database('configuracoes').where({clienteIdConf: entidade}).update({tentativas_login: process.env.LIMITE_TENTATIVAS_LOGIN})

        logger("SERVIDOR:redifinirSenha").info(`Codigo de segurança verificado`)
        const rs = response("sucesso", 202, 'Codigo de segurança verificado!','json',{
          info: {codigo_entidade: entidade},
          logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SECURITYCODEUSED" , tabela: "cliente", informacao: {tempo_de_vida_codigo_seguranca: codigo[0].tempo_de_vida_codigo_seguranca, codigo_seguranca}, entidade: entidade}
        });
        return rs
      }

      logger("SERVIDOR:redifinirSenha").info(`Codigo de segurança invalido. Coloque a sequencia correctamente`)
      const rs = response("erro", 406, 'Codigo de segurança invalido. Coloque a sequencia correctamente');
      return rs
        
      
    }else{
      logger("SERVIDOR:redifinirSenha").info(`Codigo de segurança invalido. Por favor verifique`)
      const rs = response("erro", 406, 'Codigo de segurança invalido. Por favor verifique');
      return rs 
    }
       
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:redifinirSenha").error(`Erro ao redifinir a senhaCliente ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
     
}

module.exports.postClientes = async function(dados, req) {

    try {

        logger("SERVIDOR:postClientes").debug(`Verificar o cliente por emailCliente`)
        const resultEnt  = await database('cliente')
        .where({emailCliente: dados?.emailCliente})
        
        if(resultEnt.length > 0 ){
          logger("SERVIDOR:postClientes").info(`Email usado por outra entidade`)
          const rs = response("erro", 409, "Email usado por outra entidade");
          return rs
        }
        
        logger("SERVIDOR:postClientes").error(`Verificar cliente pelo serviço Pagamento por referência`)
        const resultEmail  = await database('cliente')
        .where({telefoneCliente: dados?.telefoneCliente})
        
        if(resultEmail.length > 0 ){
          logger("SERVIDOR:postClientes").info(`Telefone usado por outra entidade`)
          const rs = response("erro", 409, "Telefone usado por outra entidade");
          return rs
        }

        logger("SERVIDOR:postClientes").error(`Verificar cliente pelo serviço Pagamento por referência`)
        const resultNif  = await database('cliente')
        .where({nifCliente: dados?.nifCliente})
        
        if(resultNif.length > 0 ){
          logger("SERVIDOR:postClientes").info(`nifCliente usado por outra entidade`)
          const rs = response("erro", 409, "nifCliente usado por outra entidade");
          return rs
        }

        logger("SERVIDOR:cliente").debug(`A cadastrar o cliente`) 
        await database('cliente').insert({...dados, nomeCliente: dados.nomeCliente.toUpperCase()})
        
        const notification = {
              mensagem: "yup.string().required()",
              para: "saudacaoDeCadastro",
              efeito: { 
                empresa: dados.nomeCliente, 
                email: dados.emailCliente
              }, 
              informacao: {}, 
              canal: 'email',
              opcional: 'email'
          }
        
        
        logger("SERVIDOR:cliente").info(`Entidade criada com sucesso`)
        const rs = response("sucesso", 201, "Empresa criada com sucesso","json",{
          logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "cliente", informacao: dados, entidade: "01157"},
          info: dados,
          notification
        });

        return rs
      
    } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:cliente").error(`Erro ao cadastrar o cliente ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
    }
    
}

module.exports.activarPorCodigo = async function({codigo_confirmacao}, req) {
  
  try {

      const verificar = await database("configuracoes")
      .where({codigo_confirmacao})

      if(!verificar.length){
        logger("SERVIDOR:activarPorLInk").info("Codigo verificado incorrecto")
        const rs = response("erro", 409, "Codigo verificado incorrecto");
        return rs    
      }

      logger("SERVIDOR:activarPorLInk").info(`Codigo verificado com sucesso`)
      await database("cliente").where({clienteId:verificar[0].clienteIdConf}).update({bloqueio:'0'})
      await database('configuracoes').andWhere({clienteIdConf: verificar[0].clienteIdConf}).update({codigo_confirmacao:"000000"}) 
      const [resultado] = await database("cliente").where({clienteId:verificar[0].clienteIdConf})
      const rs = response("sucesso", 202, {
        cliente: verificar[0].clienteIdConf,
        empresa: resultado.empresaId,
        pin_activacao: codigo_confirmacao,
        hash: resultado.endereco_mac_unico,
        text: "Codigo verificado com sucesso"
      },{
      info:{
        cliente: verificar[0].clienteIdConf,
        empresa: resultado.empresaId,
        pin_activacao: codigo_confirmacao,
        text: "Codigo verificado com sucesso"
      }});
      return rs
        
  } catch (erro) {
    console.log(erro.message)
    logger("SERVIDOR:activarPorCodigo").error(`Erro ao realizar o emailCliente ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
  
  
}

module.exports.loginClientes = async function({businessPartnerId, senhaCliente}, req) {

  
  try {
        logger("SERVIDOR:loginClientes").debug("Verificar se existencia do cliente")
        const result  = await database('cliente')
        .where({businessPartnerId})
        .andWhere({bloqueio: '0'});

        
        if(result.length == 0 ){
          logger("SERVIDOR:loginClientes").info("Seu registro não foi encontrado")
          const rs = response("erro", 401, "Seu registro não foi encontrado");
        return rs

      }else{   
        
        logger("SERVIDOR:loginClientes").debug("Buscar a configuração do cliente")
        const [verifivarPagamentoTempoReal] = await database('configuracoes').where({clienteIdConf: result[0].clienteId})        

          if((verifivarPagamentoTempoReal?.servico_principal == "false")){
            logger("SERVIDOR:loginClientes").warn("Serviço de gestão inactivo")
            const rs = response("erro", 423, "Serviço de gestão inactivo");
            return rs     
          }
          
          if(bcrypt.compareSync(senhaCliente, result[0].senhaCliente)){
                            
              
              if(verifivarPagamentoTempoReal?.tentativas_login > 0){

                  const login = new Date().toISOString().split('.')[0].replace('T',' ')
              
                  logger("SERVIDOR:loginClientes").debug("Actualizar o hash de login")
                  await database('cliente')
                  .where({businessPartnerId}).update({ultimo_login: login, endereco_mac_unico: uuidv4()})  
                  
                  logger("SERVIDOR:loginClientes").debug("Buscar dados do cliente")
                  const [resultNew]  = await database('cliente').select("nomeCliente AS empresa","endereco_mac_unico AS hash", "clienteId AS entidade","ultimo_login AS login", "novo_cliente AS novo_cliente", "empresaId AS empresa")
                  .join("configuracoes","configuracoes.clienteIdConf","=","cliente.clienteId")
                  .where({businessPartnerId})
                  

                  logger("SERVIDOR:loginClientes").debug("Repor os numeros de tentativas")
                  await database('configuracoes').where({clienteIdConf: resultNew.entidade}).update({tentativas_login: process.env.LIMITE_TENTATIVAS_LOGIN})

                  const codigo_confirmacao = String(Math.random()).replaceAll(".","").substr(0,6);
                  await database("configuracoes").where({clienteIdConf: resultNew.entidade}).update({codigo_confirmacao})

                  const [info] = await database("configuracoes").where({clienteIdConf: resultNew.entidade})

                  const notification = {
                        mensagem: "yup.string().required()",
                        para: "confirmacaoDeConta",
                        efeito: { 
                          empresa: result[0].nomeCliente, 
                          email: result[0].emailCliente, 
                          codigo_seguranca: codigo_confirmacao
                        }, 
                        informacao: {}, 
                        canal: 'email',
                        opcional: 'email'
                    }


                  logger("SERVIDOR:loginClientes").info("Cliente logado com sucesso") 
                  const rs = response("sucesso", 202, {text: `Autenticação feita com sucesso`,hash:resultNew.hash, ultimo_login: resultNew.login, novo_cliente: resultNew.novo_cliente, entidade: resultNew.entidade, empresa: resultNew.empresa},'json',{
                    info,
                    notification
                  });
                    

                  return rs

              }else{
                  logger("SERVIDOR:loginClientes").debug("Buscar dados do cliente")
                  const [resultNew]  = await database('cliente').select("businessPartnerId", "clienteId AS entidade")
                  .where({businessPartnerId})
                  

                  logger("SERVIDOR:loginClientes").warn("Cliente bloqueiado")
                  const rs = response("erro", 403, "Seu perfil se encontra bloqueado. Por favor entre em contacto com administrador", "json",{
                    logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "LOGINFORCE", tabela: "cliente", informacao: businessPartnerId, entidade: resultNew.entidade}
                  });
                  return rs

                }

          } else{

              logger("SERVIDOR:loginClientes").debug("Buscar a configuração de tentativas do cliente")
              const tentativas = await database('configuracoes').where({clienteIdConf: result[0].clienteId})
              let count = tentativas[0]?.tentativas_login - 1

              
              if(tentativas[0]?.tentativas_login > 0){
                await database('configuracoes').where({clienteIdConf: result[0].clienteId}).update({tentativas_login: count}) 
                logger("SERVIDOR:loginClientes").info("Autenticação incorrecta")
                const rs = response("erro", 401, "Autenticação incorrecta");
                return rs 
                
              }else{
                logger("SERVIDOR:loginClientes").debug("Buscar dados do cliente")
                const [resultNew]  = await database('cliente').select("clienteId AS entidade","nomeCliente AS empresa")
                .where({businessPartnerId})
                

                logger("SERVIDOR:loginClientes").warn("Cliente bloqueiado")
                const rs = response("erro", 403, "Seu perfil se encontra bloqueado. Por favor entre em contacto com administrador", "json",{
                  notification: {efeito: {empresa: resultNew.empresa, businessPartnerId}, para:"bloququeioDeConta", mensagem: 'null', canal:"emailCliente"},
                  logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "BLOCK", tabela: "cliente", informacao: businessPartnerId, entidade: resultNew.entidade}
                });
                return rs
                
              }
            
          }   
      }   
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:loginClientes").error(`Erro ao realizar o login ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
  
}

module.exports.logoutClientes = async function(clienteId, req) {
  
  try {
        logger("SERVIDOR:logoutClientes").debug("Buscar a configuração do cliente")
        const [verifivarPagamentoTempoReal] = await database('configuracoes').where({clienteIdConf: clienteId})        

        if((verifivarPagamentoTempoReal?.servico_principal == "false")){
          logger("SERVIDOR:loginClientes").warn("Serviço de gestão inactivo")
          const rs = response("erro", 423, "Serviço de gestão inactivo");
          return rs     
        }

        logger("SERVIDOR:logoutClientes").debug("Buscar dados do cliente")
        const result  = await database('cliente')
        .where({clienteId})
        
        if(result.length > 0){

          logger("SERVIDOR:logoutClientes").debug("Actualizar o hash de login para zeros")
          await database('cliente').where({clienteId}).update({endereco_mac_unico: "00000000000000000000"})
          await database('configuracoes').where({clienteIdConf: clienteId}).update({codigo_confirmacao: "000000"})

          logger("SERVIDOR:logoutClientes").info("Logout feito com sucesso")
          const rs = response("sucesso", 202, 'Logout feito com sucesso',"json", {
            logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "LOGOUT" , tabela: "cliente", informacao: result[0], entidade: result[0].clienteId}
          });
          return rs

        }else{
          logger("SERVIDOR:logoutClientes").info('Cliente desconhecido')
          const rs = response("erro", 406, 'Cliente desconhecido');
          return rs
          
        } 
        
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:logoutClientes").error(`Erro ao realizar o logout cliente ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
  
}
 
module.exports.comunicarEmail = async function({emailCliente, assunto, html}, req) {
  
  try {

      const filtered = [];

      if(emailCliente.length){
        for( const i of emailCliente){
          logger("SERVIDOR:comunicarEmail").debug(`Verificar a existencia do emailCliente nos cliente`)
          const result  = await database('cliente')
          .where({emailCliente:i})

          if(result.length) filtered.push(i)
        }
        
      }

      if(!filtered.length){
        logger("SERVIDOR:comunicarEmail").info("Sem emailCliente para enviar")
        const rs = response("erro", 409, "Sem emailCliente para enviar");
        return rs    
      }

      logger("SERVIDOR:comunicarEmail").info(`Email enviados com sucesso`)
      const rs = response("sucesso", 201, "Email enviados com sucesso", "json", {
        notification: {efeito: {assunto, filtered, html}, para:"comunicarEmail", mensagem: 'null', canal:"emailCliente"},
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "cliente", informacao: {emailCliente, assunto, html}, entidade: "01157"}
      });
      return rs
        
  } catch (erro) {
    console.log(erro.message)
    logger("SERVIDOR:comunicarEmail").error(`Erro ao realizar o emailCliente ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
  
}

module.exports.patchClientes = async function(clienteId, dados, req) { 
  try {

    logger("SERVIDOR:patchClientes").debug(`Verificar se é um  cliente do serviço GPO`)
    const clienteVerify = await database('cliente').where({clienteId})
    let entidade = ""

    if(!clienteVerify.length){
      logger("SERVIDOR:patchClientes").info("Cliente não foi encontrado")
      const rs = response("erro", 409, "Cliente não foi encontrado");
      return rs    
    }

    if(clienteVerify.length){
      clienteId = clienteVerify[0].clienteId
      entidade = clienteVerify[0].clienteId
    }


    // const actualizado_em = new Date().toISOString().replace('T',' ').substr(0,19)

    logger("SERVIDOR:patchClientes").debug(`Actualizado o cliente`)
    await database('cliente').where({clienteId}).update({...dados})

    logger("SERVIDOR:patchClientes").info(`Cliente actualizado com sucesso`)
    const rs = response("sucesso", 202, "Cliente actualizado com sucesso", "json", {
      entidadeGPO: clienteVerify[0]?.gpo_numero_comerciante,
      logs: req && {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "cliente", informacao: {...dados, clienteId}, entidade: "01157"}
    });
    return rs
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:patchClientes").error(`Erro ao buscar cliente ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
    
}

module.exports.patchClientesRedifinirSenha = async function(clienteId, dados, req) { 
  try {

      logger("SERVIDOR:patchClientesRedifinirSenha").debug(`Verificar se é um  cliente do serviço GPO`)
      const clienteVerify = await database('cliente').where({clienteId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:patchClientesRedifinirSenha").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        clienteId = clienteVerify[0].clienteId
        entidade = clienteVerify[0].clienteId
      }

      // const actualizado_em = new Date().toISOString().replace('T',' ').substr(0,19)
      
      logger("SERVIDOR:patchClientesRedifinirSenha").debug(`Actualizado o cliente`)
      await database('cliente').where({clienteId}).update({...dados})

      logger("SERVIDOR:patchClientesRedifinirSenha").info(`Palavra-passe do cliente actualizada com sucesso`)
      const rs = response("sucesso", 202, "Palavra-passe do cliente actualizada com sucesso", "json", {
        logs: req && {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "cliente", informacao: {...dados, clienteId}, entidade: "01157"}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchClientesRedifinirSenha").error(`Erro ao actualizar da senhaCliente ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchClientesTrocarSenhaPadrao = async function(clienteId, dados, req) { 
  try {

      logger("SERVIDOR:patchClientesTrocarSenhaPadrao").debug(`Verificar se é um  cliente do serviço GPO`)
      const clienteVerify = await database('cliente').where({clienteId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:patchClientesTrocarSenhaPadrao").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        clienteId = clienteVerify[0].clienteId
        entidade = clienteVerify[0].clienteId
      }


      // const actualizado_em = new Date().toISOString().replace('T',' ').substr(0,19)
      
      logger("SERVIDOR:patchClientesTrocarSenhaPadrao").debug(`Actualizado o cliente`)
      await database('cliente').where({clienteId}).update({...dados})

      logger("SERVIDOR:patchClientesTrocarSenhaPadrao").info(`Palavra-passe padrão do cliente actualizada com sucesso`)
      const rs = response("sucesso", 202, "Palavra-passe padrão do cliente actualizada com sucesso", "json", {
        logs: req && {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "cliente", informacao: {...dados, clienteId}, entidade: "01157"}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchClientesTrocarSenhaPadrao").error(`Erro ao actualizar da senhaCliente padrão ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchClientesVerificarSenhaActual = async function(clienteId, senha_actual, req) { 
  try {

      logger("SERVIDOR:patchClientesVerificarSenhaActual").debug(`Verificar se é um  cliente do serviço GPO`)
      const clienteVerify = await database('cliente').where({clienteId}).orWhere({gpo_numero_comerciante: clienteId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:patchClientesVerificarSenhaActual").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        clienteId = clienteVerify[0].clienteId
        entidade = clienteVerify[0].clienteId
      }

      if(bcrypt.compareSync(senha_actual, clienteVerify[0].senhaCliente)){

        logger("SERVIDOR:patchClientesVerificarSenhaActual").info(`Palavra-passe actual do cliente verificada com sucesso`)
        const rs = response("sucesso", 202, "Palavra-passe actual do cliente verificada com sucesso");
        return rs

      }else {

        logger("SERVIDOR:patchClientesVerificarSenhaActual").info("Cliente não foi encontrado")
        const rs = response("erro", 401, "Palavra-passe incorrecta");
        return rs

      }
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchClientesVerificarSenhaActual").error(`Erro ao actualizar da senhaCliente padrão ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchClientesAlterarSenha = async function(clienteId, senha_actual, senhaCliente, req) { 
  try {

      logger("SERVIDOR:patchClientesAlterarSenha").debug(`Verificar se é um  cliente`)
      const clienteVerify = await database('cliente').where({clienteId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:patchClientesAlterarSenha").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        entidade = clienteVerify[0].clienteId
      }


      if(bcrypt.compareSync(senha_actual, clienteVerify[0].senhaCliente)){

        // const cliente_update = new Date().toISOString().replace('T',' ').substr(0,19)
      
        logger("SERVIDOR:patchClientesRedifinirSenha").debug(`Actualizado o cliente`)
        await database('cliente').where({clienteId}).update({senhaCliente})

        logger("SERVIDOR:patchClientesAlterarSenha").info(`Palavra-passe actual do cliente verificada com sucesso`)
        const rs = response("sucesso", 202, "Palavra-passe actual do cliente verificada com sucesso");
        return rs

      }else {

        logger("SERVIDOR:patchClientesAlterarSenha").info("Cliente não foi encontrado")
        const rs = response("erro", 401, "Palavra-passe incorrecta");
        return rs

      }
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchClientesAlterarSenha").error(`Erro ao actualizar da senhaCliente padrão ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchClientesFoto = async function(clienteId, dados, req) {
  try {

      logger("SERVIDOR:mudarFotoClientes").error(`Erro ao mudar a foto do cliente`)
      const actualizado_em = new Date().toISOString().replace('T',' ').substr(0,19)
      const cliente = await database('cliente').where({clienteId})

      if(!cliente.length){
        logger("SERVIDOR:patchClientesBloquear").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:mudarFotoClientes").error(`Erro ao mudar a foto do cliente`)
      await database('cliente').where({clienteId}).update({...dados, actualizado_em})

      logger("SERVIDOR:mudarFotoClientes").error(`Erro ao mudar a foto do cliente`)
      const rs = response("sucesso", 202, "Logo da entidade actualizado com sucesso", "json", {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "PATCH" , tabela: "cliente", informacao: {...dados, clienteId}, entidade: cliente[0].clienteId}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:mudarFotoClientes").error(`Erro ao mudar a foto do cliente ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchClientesArquivoContrato = async function(clienteId, dados, req) {
  try {

      logger("SERVIDOR:patchClientesArquivoContrato").error(`Erro ao mudar a foto do cliente `)
      const actualizado_em = new Date()
      .toISOString()
      .replace('T',' ')
      .substr(0,19)
      const cliente = await database('cliente').where({clienteId})

      if(!cliente.length){
        logger("SERVIDOR:patchClientesArquivoContrato").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:patchClientesArquivoContrato").error(`Erro ao mudar a foto do cliente `)
      await database('cliente').where({clienteId}).update({...dados, actualizado_em})
      
      logger("SERVIDOR:patchClientesArquivoContrato").error(`Erro ao mudar a foto do cliente `)
      const rs = response("sucesso", 202, "Arquivo de contrato da entidade actualizado com sucesso", "json", {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "PATCH" , tabela: "cliente", informacao: {...dados, clienteId}, entidade: cliente[0].clienteId}
      });
      return rs

  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchClientesArquivoContrato").error(`Erro ao mudar o arquivo de contrato do cliente ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchClientesBloquear = async function(clienteId, req) {
  try {

      const actualizado_em = new Date()
      .toISOString()
      .replace('T',' ')
      .substr(0,19)

      logger("SERVIDOR:patchClientesBloquear").info(`Verificar o cliente pelo Id ${clienteId}`)
      const cliente = await database('cliente').where({clienteId})

      if(!cliente.length){
        logger("SERVIDOR:patchClientesBloquear").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:patchClientesBloquear").info(`Actualizando para bloquear`)
      await database('cliente').where({clienteId}).update({bloqueio:'0', actualizado_em}) 

      logger("SERVIDOR:patchClientesBloquear").info(`Cliente bloqueado com sucesso`)
      const rs = response("sucesso", 202, "Cliente bloqueado com sucesso", "json", {
        notification: {efeito: {empresa: cliente[0].nomeCliente, emailCliente: cliente[0].emailCliente}, para:"bloququeioDeContaADM", mensagem: 'null', canal:"emailCliente"},
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DESATIVE" , tabela: "cliente", informacao: {entidade: cliente[0].clienteId, clienteId}, entidade: cliente[0].clienteId}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchClientesBloquear").error(`Erro ao bloquear o cliente ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchClientesDesbloquear = async function(clienteId, req) {
  try {

      const actualizado_em = new Date()
      .toISOString()
      .replace('T',' ')
      .substr(0,19)

      logger("SERVIDOR:patchClientesDesbloquear").info(`Verificar o cliente pelo Id ${clienteId}`)
      const cliente = await database('cliente').where({clienteId})

      if(!cliente.length){
        logger("SERVIDOR:patchClientesDesbloquear").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:patchClientesDesbloquear").info(`Actualizando para desbloquear`)
      await database('cliente').where({clienteId}).update({bloqueio:'1', actualizado_em})
      
      logger("SERVIDOR:patchClientesDesbloquear").info(`Cliente desbloqueado com sucesso`)
      const rs = response("sucesso", 202, "Cliente desbloqueado com sucesso", "json", {
        notification: {efeito: {empresa: cliente[0].nomeCliente, emailCliente: cliente[0].emailCliente}, para:"desbloququeioDeContaADM", mensagem: 'null', canal:"emailCliente"},
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "ACTIVE" , tabela: "cliente", informacao: {entidade: cliente[0].clienteId, clienteId}, entidade: cliente[0].clienteId}
      });
      return rs
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:patchClientesDesbloquear").error(`Erro ao desbloquear o cliente ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
    
}

module.exports.configurarReporClientes = async function(clienteId,  tentativas_login , req) {
  try {

      logger("SERVIDOR:configurarReporClientes").debug(`Verificar a existencia do cliente`)
      const cliente = await database('cliente').where({clienteId})

      if(!cliente.length){
        logger("SERVIDOR:configurarReporClientes").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:configurarReporClientes").debug(`Actualizar os numeros de tentativas de login no padrão`)
      await database('configuracoes').where({clienteIdConf: cliente[0].clienteId}).update({tentativas_login: process.env.LIMITE_TENTATIVAS_LOGIN})

      logger("SERVIDOR:configurarReporClientes").debug(`Actualizar o cliente para o estado de novo cliente`)
      await database('cliente').where({clienteId}).update({novo_cliente:"1"})

      logger("SERVIDOR:configurarReporClientes").info(`A conta da entidade foi reposta com sucesso`)
      const rs = response("sucesso", 202, "A conta da entidade foi reposta com sucesso", "json", {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "PATH", tabela: "CONFIGURACOES", informacao: {tentativas_login, entidade: cliente[0].clienteId, clienteId}, entidade: cliente[0].clienteId}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:configurarReporClientes").error(`Erro ao repor o cliente ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.deleteClientes = async function(clienteId, req) { 
  try {

      logger("SERVIDOR:deleteClientes").debug(`Verificar se o cliente é do serviço GPO`)
      const clienteVerify = await database('cliente').where({clienteId}).orWhere({gpo_numero_comerciante:clienteId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:deleteClientes").info("Cliente não foi encontrado")
        const rs = response("erro", 409, "Cliente não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        clienteId = clienteVerify[0].clienteId
        entidade = clienteVerify[0].clienteId
      }   

      logger("SERVIDOR:deleteClientes").debug(`Verificar a existência do cliente`)
      const cliente = await database('cliente').where({clienteId})
      
      logger("SERVIDOR:deleteClientes").debug(`Verificar se o cliente tem referências geradas e usadas`)
      const clientesReferences = await database('clienteIds').where({entidade_cliente: cliente[0].clienteId});
      
      if(clientesReferences.length){
        logger("SERVIDOR:deleteClientes").info(`Cliente não exluido. Tem clienteIds geradas`)
        const rs = response("erro", 409, "Cliente não exluido. Tem clienteIds geradas");
        return rs
      }

      logger("SERVIDOR:deleteClientes").debug(`Verificar se o cliente tem historico de pagamentos`)
      const clientesPagamentos = await database('pagamentos').where({clienteId: cliente[0].clienteId});

      if(clientesPagamentos.length){
        logger("SERVIDOR:deleteClientes").info(`Cliente não exluido. Tem pagamentos feitos`)
        const rs = response("erro", 409, "Cliente não exluido. Tem pagamentos feitos");
        return rs
      }

      logger("SERVIDOR:deleteClientes").debug(`Á apagar o cliente`)
      await database('cliente').where({clienteId}).del() 

      logger("SERVIDOR:deleteClientes").info("Cliente exluido com sucesso")
      const rs = response("sucesso", 202, "Cliente exluido com sucesso", "json", {
        gpo_comerciante_hash: cliente[0].gpo_comerciante_hash,      
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DELETE" , tabela: "cliente", informacao: {entidade: cliente[0].clienteId, clienteId}, entidade: cliente[0].clienteId}
      });
      return rs

  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:cliente").error(`Erro ao deletar o cliente  ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}