const database = require('../config/database')
const bcrypt = require('bcryptjs');
const { v4: uuidv4 } = require('uuid');
const path = require("path");
const response = require("../constants/response");
const logger = require('../services/loggerService');
const paginationRecords = require("../helpers/paginationRecords")
const { clientesTruesFilteres } = require('../helpers/filterResponseSQL');
require("dotenv").config({ path: path.resolve(path.join(__dirname,'../','.env')) });

module.exports.getEmpresas = async function(pagina, limite, nomeEmpresa, nif, emailEmpresa, entidade, responsavel) {
  try {
      
      logger("SERVIDOR:Empresas").debug("Selecionar da base de dados")

      const empresa = await database('empresa')
      .join('usuarios',"usuarios.id_usuarios","=","empresa.criado_por")
      .join('configuracoesEmpresa',"configuracoesEmpresa.empresaIdConf","=","empresa.empresaId")
      .whereLike("nomeEmpresa",`%${String(nomeEmpresa).toUpperCase()}%`)
      .whereLike("nif",`%${nif}%`)
      .whereLike("emailEmpresa",`%${emailEmpresa}%`)
      .whereLike("empresaId",`%${entidade}%`)
      .whereLike("responsavel",`%${responsavel}%`)
      .orderBy('empresaId','DESC')

      const {registros} = paginationRecords(empresa, pagina, limite)

      logger("Empresas").debug(`Buscar todos empresa no banco de dados com limite de ${registros.limite} na pagina ${registros.count} de registros`);
      const clientesLimite = await database('empresa')
      .join('usuarios',"usuarios.id_usuarios","=","empresa.criado_por")
      .join('configuracoesEmpresa',"configuracoesEmpresa.empresaIdConf","=","empresa.empresaId")
      .whereLike("nomeEmpresa",`%${String(nomeEmpresa).toUpperCase()}%`)
      .whereLike("nif",`%${nif}%`)
      .whereLike("emailEmpresa",`%${emailEmpresa}%`)
      .whereLike("empresaId",`%${entidade}%`)
      .whereLike("responsavel",`%${responsavel}%`)
      .limit(registros.limite)
      .offset(registros.count)
      .orderBy('empresaId','DESC')

      const filtered = clientesTruesFilteres(clientesLimite)

      registros.total_apresentados = clientesLimite.length
      registros.nomeEmpresa = nomeEmpresa
      registros.nif = nif
      registros.emailEmpresa = emailEmpresa
      registros.entidade = entidade
      registros.responsavel = responsavel

      logger("SERVIDOR:Empresas").info("Respondeu a solicitação")
      const rs = response("sucesso", 200, filtered, "json", { registros });
      return rs


  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:Empresas").error(`Erro ao buscar empresa ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.getEmpresasID = async function(empresaId) {
  try {

      logger("SERVIDOR:EmpresasId").debug("Selecionar da base de dados")
      const [empresa] = await database('empresa')
      .join('usuarios',"usuarios.id_usuarios","=","empresa.criado_por")
      .join('configuracoesEmpresa',"configuracoesEmpresa.empresaIdConf","=","empresa.empresaId")
      .where({empresaId})
      .orderBy('empresaId','DESC')
    
      logger("SERVIDOR:EmpresasId").info("Respondeu a solicitação")
      const rs = response("sucesso", 200, empresa || {});          
      return rs

  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:EmpresasId").error(`Erro ao buscar empresa por ID ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.getEmpresasEntidade = async function(empresaId) {
  try {

      logger("SERVIDOR:getEmpresasEntidade").debug("Á buscar os dados")
      const empresa = await database('empresa')
      .join('usuarios',"usuarios.id_usuarios","=","empresa.criado_por")
      .join('configuracoesEmpresa',"configuracoesEmpresa.empresaIdConf","=","empresa.empresaId")
      .where({empresaId})
      .orderBy('empresaId','DESC')  
    
      logger("SERVIDOR:getEmpresasEntidade").info("Respondeu a solicitação")
      const rs = response("sucesso", 200, empresa, "json");          
      return rs

  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:getEmpresasEntidade").error(`Erro ao buscar empresa por entidade ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.getEmpresasHash = async function(hash) {
  try {

      logger("SERVIDOR:EmpresasHash").debug("Verificar se existencia do empresa")
      const result  = await database('empresa')
      .where({endereco_mac_unico_empresa:hash})
      .andWhere({bloqueioEmpresa: '0'});

      if(result.length == 0 ){
        logger("SERVIDOR:EmpresasHash").info("Empresa está bloqueado")
        const rs = response("erro", 403, "Empresa está bloqueado");
        return rs

      }else{

        logger("SERVIDOR:EmpresasHash").debug("Buscar a configuração do empresa")
        const [verifivarPagamentoTempoReal] = await database('configuracoesEmpresa').where({empresaIdConf: result[0].empresaId}) 

        console.log(verifivarPagamentoTempoReal?.servico_principal)

        if((verifivarPagamentoTempoReal?.servico_principal == "false")){
          logger("SERVIDOR:loginEmpresas").warn("Serviço de gestão inactivo")
          const rs = response("erro", 423, "Serviço de gestão inactivo");
          return rs     
        }

        if(verifivarPagamentoTempoReal?.tentativas_login > 0){

          logger("SERVIDOR:EmpresasHash").debug("Selecionar da base de dados")
          const [empresa] = await database('empresa')
          .join('configuracoesEmpresa',"configuracoesEmpresa.empresaIdConf","=","empresa.empresaId")
          .where({endereco_mac_unico_empresa:hash})
          .orderBy('empresaId','DESC')  
        
          logger("SERVIDOR:EmpresasHash").info("Respondeu a solicitação")
          const rs = response("sucesso", 200, empresa, "json");          
          return rs

        }else{
          logger("SERVIDOR:EmpresasHash").warn("Empresa está bloqueiado")
          const rs = response("erro", 403, "Seu perfil se encontra bloqueado. Por favor entre em contacto com administrador");
          return rs
        }

      }

  } catch (erro) { 
      console.log(erro)
      logger("SERVIDOR:EmpresasHash").error(`Erro ao buscar empresa por hash ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.getEmpresasEmail = async function(emailEmpresa) { 
  try {

      logger("SERVIDOR:EmpresasEmail").debug("Verificar se existencia do empresa")
      const empresa = await database('empresa')
      .join('usuarios',"usuarios.id_usuarios","=","empresa.criado_por")
      .join('configuracoesEmpresa',"configuracoesEmpresa.empresaIdConf","=","empresa.empresaId")
      .where({emailEmpresa})
      .orderBy('empresaId','DESC')
    
      logger("SERVIDOR:EmpresasEmail").info("Respondeu a solicitação")
      const rs = response("sucesso", 200, empresa, "json");          
      return rs
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:EmpresasEmail").error(`Erro ao buscar empresa ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
    
}

module.exports.recuperarSenha = async function(emailEmpresa, canal, req) {
  try {

    logger("SERVIDOR:recuperarSenha").debug(`Verificar a existencia do empresa pelo emailEmpresa ${emailEmpresa}`)
    const empresa = await database('empresa')
    .where({emailEmpresa})
    .orWhere({telefoneEmpresa: emailEmpresa})
    .orderBy('empresaId','DESC')
    
    logger("SERVIDOR:recuperarSenha").debug(`Gerando um codigo de 6 digitos`)
    let codigo_seguranca = String(Math.random()).replaceAll(".","").substr(0,6);
    
    var now = new Date();
    var time = now.getTime();
    var expireTime = time + 1000*60*10;
    
    let tempo_de_vida_codigo_seguranca = expireTime;
    
    logger("SERVIDOR:recuperarSenha").info(`Codigo gerado: ${codigo_seguranca} para ${tempo_de_vida_codigo_seguranca} de vida util`)
    if(empresa.length > 0) {

      logger("SERVIDOR:recuperarSenha").debug(`Verficar duplicidade de codigo`)
      const codigounico = await database('configuracoesEmpresa').where({codigo_seguranca})
      
      logger("SERVIDOR:recuperarSenha").debug(`Gerar outro codigo`)
      while(codigounico.length > 0) {
          codigo_seguranca = String(Math.random()).replaceAll(".","").substr(0,6);
      }
      
      logger("SERVIDOR:recuperarSenha").debug(`Actualizando e gravando na base de dados`)
      await database('configuracoesEmpresa').where({empresaIdConf: empresa[0].empresaId}).update({codigo_seguranca, tempo_de_vida_codigo_seguranca})
      
      
        if(canal === "Whatsapp") {
            logger("SERVIDOR:recuperarSenha").info(`Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`)
            const rs = response("sucesso", 202, `Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`, 'json',{
              notification: {efeito: {contacto: empresa[0].contacto, empresaId: empresa[0].empresaId, emailEmpresa: empresa[0].emailEmpresa, codigo_seguranca }, para:"codigoSeguranca", mensagem: 'null', canal:"whatsapp"},
              info: {entidade:empresa[0].empresaId, tempo_de_vida_codigo_seguranca},
              logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SENDSECURITYCODE" , tabela: "CLIENTES", informacao: {emailEmpresa, tempo_de_vida_codigo_seguranca, codigo_seguranca, canal}, entidade: empresa[0].empresaId}
            });
            return rs
        }
        else if(canal === "E-mail") {

            logger("SERVIDOR:recuperarSenha").info(`Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`)
            const rs = response("sucesso", 202, `Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`, 'json',{
              notification: {efeito: {empresa: empresa[0].nomeEmpresa, emailEmpresa, codigo_seguranca }, para:"codigoSeguranca", mensagem: 'null', canal:"emailEmpresa"},
              info: {entidade:empresa[0].empresaId, tempo_de_vida_codigo_seguranca},
              logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SENDSECURITYCODE" , tabela: "CLIENTES", informacao: {emailEmpresa, tempo_de_vida_codigo_seguranca, codigo_seguranca, canal}, entidade: empresa[0].empresaId}
            });
            return rs
        }
        else if(canal.toUpperCase() === "SMS") {
            
            logger("SERVIDOR:recuperarSenha").info(`Caro(a) empresa, realizou em pedido de alteração de sua senhaEmpresa. O codigo é:  ${codigo_seguranca}`)
            const mensagem = `Caro(a) empresa, realizou em pedido de alteração de sua senhaEmpresa. O codigo é:  ${codigo_seguranca}`;
            const rs = response("sucesso", 202, `Enviamos um codigo de segurança para o seu ${canal}. Por favor verifique`, 'json',{
              notification: {efeito: {...empresa[0]}, para:"RECUPERACAO", mensagem, canal:"sms"},
              info: {entidade:empresa[0].empresaId, tempo_de_vida_codigo_seguranca},
              logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SENDSECURITYCODE" , tabela: "CLIENTES", informacao: {emailEmpresa, tempo_de_vida_codigo_seguranca, codigo_seguranca, canal}, entidade: empresa[0].empresaId}
            });
            return rs
        }
        else{
            logger("SERVIDOR:recuperarSenha").info(`Não conseguimos enviar o codigo de segurança para o seu ${canal}`)
            const rs = response("erro", 406, `Não conseguimos enviar o codigo de segurança para o seu ${canal}`);
            return rs 
        }
        
    }else{
    
      logger("SERVIDOR:recuperarSenha").info(`Não conseguimos seu E-mail nos nossos registros`)
      const rs = response("erro", 409, 'Não conseguimos seu E-mail nos nossos registros');
      return rs             
      
    }    
    
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:recuperarSenha").error(`Erro ao recuperar a senhaEmpresa ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs 
  }
    
}

module.exports.redifinirSenha = async function(codigo_seguranca, entidade, req) {
  try {

    logger("SERVIDOR:redifinirSenha").debug(`Verificar o codigo de segurança`)
    const codigo = await database('configuracoesEmpresa').where({codigo_seguranca}).andWhere({empresaIdConf: entidade})
    
    if(codigo.length > 0) {
    
      var now = new Date();
      var time = now.getTime();
      
      if(time > codigo[0].tempo_de_vida_codigo_seguranca){
        logger("SERVIDOR:redifinirSenha").info(`Codigo de segurança expirado. Por favor repita os passos novamente`)
        const rs = response("erro", 406, 'Codigo de segurança expirado. Por favor repita os passos novamente!','json',{
          logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SECURITYCODEEXPIRE" , tabela: "CLIENTES", informacao: {tempo_de_vida_codigo_seguranca: codigo[0].tempo_de_vida_codigo_seguranca, codigo_seguranca}, entidade: entidade}
        });
        return rs
      }
      

      logger("SERVIDOR:redifinirSenha").debug(`Actualizar o codigo de segurança para 0000`)
      await database('configuracoesEmpresa').andWhere({empresaIdConf: entidade}).update({codigo_seguranca:"000000"}) 
      
      if(codigo_seguranca != "0000"){
        logger("SERVIDOR:redifinirSenha").debug(`Actualizar e resetar as ${process.env.LIMITE_TENTATIVAS_LOGIN} tentativas`)
        await database('configuracoesEmpresa').where({empresaIdConf: entidade}).update({tentativas_login: process.env.LIMITE_TENTATIVAS_LOGIN})

        logger("SERVIDOR:redifinirSenha").info(`Codigo de segurança verificado`)
        const rs = response("sucesso", 202, 'Codigo de segurança verificado!','json',{
          info: {codigo_entidade: entidade},
          logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "SECURITYCODEUSED" , tabela: "CLIENTES", informacao: {tempo_de_vida_codigo_seguranca: codigo[0].tempo_de_vida_codigo_seguranca, codigo_seguranca}, entidade: entidade}
        });
        return rs
      }

      logger("SERVIDOR:redifinirSenha").info(`Codigo de segurança invalido. Coloque a sequencia correctamente`)
      const rs = response("erro", 406, 'Codigo de segurança invalido. Coloque a sequencia correctamente');
      return rs
        
      
    }else{
      logger("SERVIDOR:redifinirSenha").info(`Codigo de segurança invalido. Por favor verifique`)
      const rs = response("erro", 406, 'Codigo de segurança invalido. Por favor verifique');
      return rs 
    }
       
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:redifinirSenha").error(`Erro ao redifinir a senhaEmpresa ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
     
}

module.exports.postEmpresas = async function(dados, req) {

    try {

        logger("SERVIDOR:postEmpresas").debug(`Verificar o empresa por emailEmpresa`)
        const resultEnt  = await database('empresa')
        .where({emailEmpresa: dados?.emailEmpresa})
        
        if(resultEnt.length > 0 ){
          logger("SERVIDOR:postEmpresas").info(`Email usado por outra entidade`)
          const rs = response("erro", 409, "Email usado por outra entidade");
          return rs
        }
        
        logger("SERVIDOR:postEmpresas").error(`Verificar empresa pelo serviço Pagamento por referência`)
        const resultEmail  = await database('empresa')
        .where({telefoneEmpresa: dados?.telefoneEmpresa})
        
        if(resultEmail.length > 0 ){
          logger("SERVIDOR:postEmpresas").info(`Telefone usado por outra entidade`)
          const rs = response("erro", 409, "Telefone usado por outra entidade");
          return rs
        }

        logger("SERVIDOR:postEmpresas").error(`Verificar empresa pelo serviço Pagamento por referência`)
        const resultNif  = await database('empresa')
        .where({nifEmpresa: dados?.nifEmpresa})
        
        if(resultNif.length > 0 ){
          logger("SERVIDOR:postEmpresas").info(`NIF usado por outra entidade`)
          const rs = response("erro", 409, "NIF usado por outra entidade");
          return rs
        }

        logger("SERVIDOR:Empresas").debug(`A cadastrar o empresa`) 
        await database('empresa').insert({...dados, nomeEmpresa: dados.nomeEmpresa.toUpperCase()})
        
        
        
        logger("SERVIDOR:Empresas").info(`Entidade criada com sucesso`)
        const rs = response("sucesso", 201, "Empresa criada com sucesso","json",{
          logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "CLIENTES", informacao: dados, entidade: "01157"},
          info: dados
        });

        return rs
      
    } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:Empresas").error(`Erro ao cadastrar o empresa ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
    }
    
}

module.exports.activarPorCodigo = async function({codigo_confirmacao}, req) {
  
  try {

      const verificar = await database("configuracoesEmpresa")
      .where({codigo_confirmacao})

      if(!verificar.length){
        logger("SERVIDOR:activarPorLInk").info("Codigo verificado incorrecto")
        const rs = response("erro", 409, "Codigo verificado incorrecto");
        return rs    
      }

      logger("SERVIDOR:activarPorLInk").info(`Codigo verificado com sucesso`)
      await database("empresa").where({empresaId:verificar[0].empresaIdConf}).update({bloqueioEmpresa:'0'})
      await database('configuracoesEmpresa').andWhere({empresaIdConf: verificar[0].empresaIdConf}).update({codigo_confirmacao:"000000"}) 
      const [resultado] = await database("empresa").where({empresaId:verificar[0].empresaIdConf})
      const rs = response("sucesso", 202, {
        empresa: verificar[0].empresaIdConf,
        pin_activacao: codigo_confirmacao,
        hash: resultado.endereco_mac_unico_empresa,
        text: "Codigo verificado com sucesso"
      },{
      info:{
        empresa: verificar[0].empresaIdConf,
        pin_activacao: codigo_confirmacao,
        text: "Codigo verificado com sucesso"
      }});
      return rs
        
  } catch (erro) {
    console.log(erro.message)
    logger("SERVIDOR:activarPorCodigo").error(`Erro ao realizar o emailEmpresa ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
  
  
}

module.exports.loginEmpresas = async function({emailEmpresa, senhaEmpresa}, req) {

  
  try {
        logger("SERVIDOR:loginEmpresas").debug("Verificar se existencia do empresa")
        const result  = await database('empresa')
        .where({emailEmpresa})
        .andWhere({bloqueioEmpresa: '0'});

        
        if(result.length == 0 ){
          logger("SERVIDOR:loginEmpresas").info("Seu registro não foi encontrado")
          const rs = response("erro", 401, "Seu registro não foi encontrado");
        return rs

      }else{   
        
        logger("SERVIDOR:loginEmpresas").debug("Buscar a configuração do empresa")
        const [verifivarPagamentoTempoReal] = await database('configuracoesEmpresa').where({empresaIdConf: result[0].empresaId})        

          if((verifivarPagamentoTempoReal?.servico_principal == "false")){
            logger("SERVIDOR:loginEmpresas").warn("Serviço de gestão inactivo")
            const rs = response("erro", 423, "Serviço de gestão inactivo");
            return rs     
          }
          
          if(bcrypt.compareSync(senhaEmpresa, result[0].senhaEmpresa)){
                            
              
              if(verifivarPagamentoTempoReal?.tentativas_login > 0){

                  const login = new Date().toISOString().split('.')[0].replace('T',' ')
              
                  logger("SERVIDOR:loginEmpresas").debug("Actualizar o hash de login")
                  await database('empresa')
                  .where({emailEmpresa}).update({ultimo_login: login, endereco_mac_unico_empresa: uuidv4()})  
                  
                  logger("SERVIDOR:loginEmpresas").debug("Buscar dados do empresa")
                  const [resultNew]  = await database('empresa').select("nomeEmpresa AS empresa","endereco_mac_unico_empresa AS hash", "empresaId AS entidade","ultimo_login AS login", "nova_empresa AS nova_empresa")
                  .join("configuracoesEmpresa","configuracoesEmpresa.empresaIdConf","=","empresa.empresaId")
                  .where({emailEmpresa})
                  

                  logger("SERVIDOR:loginEmpresas").debug("Repor os numeros de tentativas")
                  await database('configuracoesEmpresa').where({empresaIdConf: resultNew.entidade}).update({tentativas_login: process.env.LIMITE_TENTATIVAS_LOGIN})

                  logger("SERVIDOR:loginEmpresas").info("Empresa logado com sucesso") 
                  const rs = response("sucesso", 202, {text: `Autenticação feita com sucesso`,hash:resultNew.hash, ultimo_login: resultNew.login, nova_empresa: resultNew.nova_empresa, entidade: resultNew.entidade}
                  
                  );

                  const codigo_confirmacao = String(Math.random()).replaceAll(".","").substr(0,6);
                  await database("configuracoesEmpresa").where({empresaIdConf: resultNew.entidade}).update({codigo_confirmacao})

                  const [info] = await database("configuracoesEmpresa").where({empresaIdConf: resultNew.entidade})

                  // const notification = {
                  //       mensagem: "yup.string().required()",
                  //       para: validacao || "confirmacaoDeConta",
                  //       efeito: { 
                  //         empresa: dados.nomeEmpresa, 
                  //         emailEmpresa: dados.emailEmpresa, 
                  //         codigo_seguranca: validacao == "confirmacaoDeContaLink" ? info.link_confirmacao : codigo_confirmacao
                  //       }, 
                  //       informacao: {}, 
                  //       canal: 'emailEmpresa',
                  //       opcional: 'emailEmpresa'
                  //   }
                    

                  return rs

              }else{
                  logger("SERVIDOR:loginEmpresas").debug("Buscar dados do empresa")
                  const [resultNew]  = await database('empresa').select("emailEmpresa", "empresaId AS entidade")
                  .where({emailEmpresa})
                  

                  logger("SERVIDOR:loginEmpresas").warn("Empresa bloqueiado")
                  const rs = response("erro", 403, "Seu perfil se encontra bloqueado. Por favor entre em contacto com administrador", "json",{
                    logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "LOGINFORCE", tabela: "CLIENTES", informacao: emailEmpresa, entidade: resultNew.entidade}
                  });
                  return rs

                }

          } else{

              logger("SERVIDOR:loginEmpresas").debug("Buscar a configuração de tentativas do empresa")
              const tentativas = await database('configuracoesEmpresa').where({empresaIdConf: result[0].empresaId})
              let count = tentativas[0]?.tentativas_login - 1

              
              if(tentativas[0]?.tentativas_login > 0){
                await database('configuracoesEmpresa').where({empresaIdConf: result[0].empresaId}).update({tentativas_login: count}) 
                logger("SERVIDOR:loginEmpresas").info("Autenticação incorrecta")
                const rs = response("erro", 401, "Autenticação incorrecta");
                return rs 
                
              }else{
                logger("SERVIDOR:loginEmpresas").debug("Buscar dados do empresa")
                const [resultNew]  = await database('empresa').select("empresaId AS entidade","nomeEmpresa AS empresa")
                .where({emailEmpresa})
                

                logger("SERVIDOR:loginEmpresas").warn("Empresa bloqueiado")
                const rs = response("erro", 403, "Seu perfil se encontra bloqueado. Por favor entre em contacto com administrador", "json",{
                  notification: {efeito: {empresa: resultNew.empresa, emailEmpresa}, para:"bloququeioDeConta", mensagem: 'null', canal:"emailEmpresa"},
                  logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "BLOCK", tabela: "CLIENTES", informacao: emailEmpresa, entidade: resultNew.entidade}
                });
                return rs
                
              }
            
          }   
      }   
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:loginEmpresas").error(`Erro ao realizar o login ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
  
}

module.exports.logoutEmpresas = async function(empresaId, req) {
  
  try {
        logger("SERVIDOR:logoutEmpresas").debug("Buscar a configuração do empresa")
        const [verifivarPagamentoTempoReal] = await database('configuracoesEmpresa').where({empresaIdConf: empresaId})        

        if((verifivarPagamentoTempoReal?.servico_principal == "false")){
          logger("SERVIDOR:loginEmpresas").warn("Serviço de gestão inactivo")
          const rs = response("erro", 423, "Serviço de gestão inactivo");
          return rs     
        }

        logger("SERVIDOR:logoutEmpresas").debug("Buscar dados do empresa")
        const result  = await database('empresa')
        .where({empresaId})
        
        if(result.length > 0){

          logger("SERVIDOR:logoutEmpresas").debug("Actualizar o hash de login para zeros")
          await database('empresa').where({empresaId}).update({endereco_mac_unico_empresa: "00000000000000000000"})
          await database('configuracoesEmpresa').where({empresaIdConf: empresaId}).update({codigo_confirmacao: "000000"})

          logger("SERVIDOR:logoutEmpresas").info("Logout feito com sucesso")
          const rs = response("sucesso", 202, 'Logout feito com sucesso',"json", {
            logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "LOGOUT" , tabela: "CLIENTES", informacao: result[0], entidade: result[0].empresaId}
          });
          return rs

        }else{
          logger("SERVIDOR:logoutEmpresas").info('Empresa desconhecido')
          const rs = response("erro", 406, 'Empresa desconhecido');
          return rs
          
        } 
        
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:logoutEmpresas").error(`Erro ao realizar o logout empresa ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
  
}
 
module.exports.comunicarEmail = async function({emailEmpresa, assunto, html}, req) {
  
  try {

      const filtered = [];

      if(emailEmpresa.length){
        for( const i of emailEmpresa){
          logger("SERVIDOR:comunicarEmail").debug(`Verificar a existencia do emailEmpresa nos empresa`)
          const result  = await database('empresa')
          .where({emailEmpresa:i})

          if(result.length) filtered.push(i)
        }
        
      }

      if(!filtered.length){
        logger("SERVIDOR:comunicarEmail").info("Sem emailEmpresa para enviar")
        const rs = response("erro", 409, "Sem emailEmpresa para enviar");
        return rs    
      }

      logger("SERVIDOR:comunicarEmail").info(`Email enviados com sucesso`)
      const rs = response("sucesso", 201, "Email enviados com sucesso", "json", {
        notification: {efeito: {assunto, filtered, html}, para:"comunicarEmail", mensagem: 'null', canal:"emailEmpresa"},
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "CLIENTES", informacao: {emailEmpresa, assunto, html}, entidade: "01157"}
      });
      return rs
        
  } catch (erro) {
    console.log(erro.message)
    logger("SERVIDOR:comunicarEmail").error(`Erro ao realizar o emailEmpresa ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
  
}

module.exports.patchEmpresas = async function(empresaId, dados, req) { 
  try {

    logger("SERVIDOR:patchEmpresas").debug(`Verificar se é um  empresa do serviço GPO`)
    const clienteVerify = await database('empresa').where({empresaId})
    let entidade = ""

    if(!clienteVerify.length){
      logger("SERVIDOR:patchEmpresas").info("Empresa não foi encontrado")
      const rs = response("erro", 409, "Empresa não foi encontrado");
      return rs    
    }

    if(clienteVerify.length){
      empresaId = clienteVerify[0].empresaId
      entidade = clienteVerify[0].empresaId
    }


    // const actualizado_em = new Date().toISOString().replace('T',' ').substr(0,19)

    logger("SERVIDOR:patchEmpresas").debug(`Actualizado o empresa`)
    await database('empresa').where({empresaId}).update({...dados})

    logger("SERVIDOR:patchEmpresas").info(`Empresa actualizado com sucesso`)
    const rs = response("sucesso", 202, "Empresa actualizado com sucesso", "json", {
      entidadeGPO: clienteVerify[0]?.gpo_numero_comerciante,
      logs: req && {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "CLIENTES", informacao: {...dados, empresaId}, entidade: "01157"}
    });
    return rs
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:patchEmpresas").error(`Erro ao buscar empresa ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
    
}

module.exports.patchEmpresasRedifinirSenha = async function(empresaId, dados, req) { 
  try {

      logger("SERVIDOR:patchEmpresasRedifinirSenha").debug(`Verificar se é um  empresa do serviço GPO`)
      const clienteVerify = await database('empresa').where({empresaId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:patchEmpresasRedifinirSenha").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        empresaId = clienteVerify[0].empresaId
        entidade = clienteVerify[0].empresaId
      }

      // const actualizado_em = new Date().toISOString().replace('T',' ').substr(0,19)
      
      logger("SERVIDOR:patchEmpresasRedifinirSenha").debug(`Actualizado o empresa`)
      await database('empresa').where({empresaId}).update({...dados})

      logger("SERVIDOR:patchEmpresasRedifinirSenha").info(`Palavra-passe do empresa actualizada com sucesso`)
      const rs = response("sucesso", 202, "Palavra-passe do empresa actualizada com sucesso", "json", {
        logs: req && {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "CLIENTES", informacao: {...dados, empresaId}, entidade: "01157"}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchEmpresasRedifinirSenha").error(`Erro ao actualizar da senhaEmpresa ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchEmpresasTrocarSenhaPadrao = async function(empresaId, dados, req) { 
  try {

      logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").debug(`Verificar se é um  empresa do serviço GPO`)
      const clienteVerify = await database('empresa').where({empresaId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        empresaId = clienteVerify[0].empresaId
        entidade = clienteVerify[0].empresaId
      }


      // const actualizado_em = new Date().toISOString().replace('T',' ').substr(0,19)
      
      logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").debug(`Actualizado o empresa`)
      await database('empresa').where({empresaId}).update({...dados})

      logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").info(`Palavra-passe padrão do empresa actualizada com sucesso`)
      const rs = response("sucesso", 202, "Palavra-passe padrão do empresa actualizada com sucesso", "json", {
        logs: req && {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "CLIENTES", informacao: {...dados, empresaId}, entidade: "01157"}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchEmpresasTrocarSenhaPadrao").error(`Erro ao actualizar da senhaEmpresa padrão ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchEmpresasVerificarSenhaActual = async function(empresaId, senha_actual, req) { 
  try {

      logger("SERVIDOR:patchEmpresasVerificarSenhaActual").debug(`Verificar se é um  empresa do serviço GPO`)
      const clienteVerify = await database('empresa').where({empresaId}).orWhere({gpo_numero_comerciante: empresaId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:patchEmpresasVerificarSenhaActual").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        empresaId = clienteVerify[0].empresaId
        entidade = clienteVerify[0].empresaId
      }

      if(bcrypt.compareSync(senha_actual, clienteVerify[0].senhaEmpresa)){

        logger("SERVIDOR:patchEmpresasVerificarSenhaActual").info(`Palavra-passe actual do empresa verificada com sucesso`)
        const rs = response("sucesso", 202, "Palavra-passe actual do empresa verificada com sucesso");
        return rs

      }else {

        logger("SERVIDOR:patchEmpresasVerificarSenhaActual").info("Empresa não foi encontrado")
        const rs = response("erro", 401, "Palavra-passe incorrecta");
        return rs

      }
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchEmpresasVerificarSenhaActual").error(`Erro ao actualizar da senhaEmpresa padrão ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchEmpresasAlterarSenha = async function(empresaId, senha_actual, senhaEmpresa, req) { 
  try {

      logger("SERVIDOR:patchEmpresasAlterarSenha").debug(`Verificar se é um  empresa`)
      const clienteVerify = await database('empresa').where({empresaId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:patchEmpresasAlterarSenha").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        entidade = clienteVerify[0].empresaId
      }


      if(bcrypt.compareSync(senha_actual, clienteVerify[0].senhaEmpresa)){

        // const cliente_update = new Date().toISOString().replace('T',' ').substr(0,19)
      
        logger("SERVIDOR:patchEmpresasRedifinirSenha").debug(`Actualizado o empresa`)
        await database('empresa').where({empresaId}).update({senhaEmpresa})

        logger("SERVIDOR:patchEmpresasAlterarSenha").info(`Palavra-passe actual do empresa verificada com sucesso`)
        const rs = response("sucesso", 202, "Palavra-passe actual do empresa verificada com sucesso");
        return rs

      }else {

        logger("SERVIDOR:patchEmpresasAlterarSenha").info("Empresa não foi encontrado")
        const rs = response("erro", 401, "Palavra-passe incorrecta");
        return rs

      }
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchEmpresasAlterarSenha").error(`Erro ao actualizar da senhaEmpresa padrão ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchEmpresasFoto = async function(empresaId, dados, req) {
  try {

      logger("SERVIDOR:mudarFotoEmpresas").error(`Erro ao mudar a foto do empresa`)
      const actualizado_em = new Date().toISOString().replace('T',' ').substr(0,19)
      const empresa = await database('empresa').where({empresaId})

      if(!empresa.length){
        logger("SERVIDOR:patchEmpresasBloquear").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:mudarFotoEmpresas").error(`Erro ao mudar a foto do empresa`)
      await database('empresa').where({empresaId}).update({...dados, actualizado_em})

      logger("SERVIDOR:mudarFotoEmpresas").error(`Erro ao mudar a foto do empresa`)
      const rs = response("sucesso", 202, "Logo da entidade actualizado com sucesso", "json", {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "PATCH" , tabela: "CLIENTES", informacao: {...dados, empresaId}, entidade: empresa[0].empresaId}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:mudarFotoEmpresas").error(`Erro ao mudar a foto do empresa ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchEmpresasArquivoContrato = async function(empresaId, dados, req) {
  try {

      logger("SERVIDOR:patchEmpresasArquivoContrato").error(`Erro ao mudar a foto do empresa `)
      const actualizado_em = new Date()
      .toISOString()
      .replace('T',' ')
      .substr(0,19)
      const empresa = await database('empresa').where({empresaId})

      if(!empresa.length){
        logger("SERVIDOR:patchEmpresasArquivoContrato").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:patchEmpresasArquivoContrato").error(`Erro ao mudar a foto do empresa `)
      await database('empresa').where({empresaId}).update({...dados, actualizado_em})
      
      logger("SERVIDOR:patchEmpresasArquivoContrato").error(`Erro ao mudar a foto do empresa `)
      const rs = response("sucesso", 202, "Arquivo de contrato da entidade actualizado com sucesso", "json", {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "PATCH" , tabela: "CLIENTES", informacao: {...dados, empresaId}, entidade: empresa[0].empresaId}
      });
      return rs

  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchEmpresasArquivoContrato").error(`Erro ao mudar o arquivo de contrato do empresa ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchEmpresasBloquear = async function(empresaId, req) {
  try {

      const actualizado_em = new Date()
      .toISOString()
      .replace('T',' ')
      .substr(0,19)

      logger("SERVIDOR:patchEmpresasBloquear").info(`Verificar o empresa pelo Id ${empresaId}`)
      const empresa = await database('empresa').where({empresaId})

      if(!empresa.length){
        logger("SERVIDOR:patchEmpresasBloquear").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:patchEmpresasBloquear").info(`Actualizando para bloquear`)
      await database('empresa').where({empresaId}).update({bloqueioEmpresa:'0', actualizado_em}) 

      logger("SERVIDOR:patchEmpresasBloquear").info(`Empresa bloqueado com sucesso`)
      const rs = response("sucesso", 202, "Empresa bloqueado com sucesso", "json", {
        notification: {efeito: {empresa: empresa[0].nomeEmpresa, emailEmpresa: empresa[0].emailEmpresa}, para:"bloququeioDeContaADM", mensagem: 'null', canal:"emailEmpresa"},
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DESATIVE" , tabela: "CLIENTES", informacao: {entidade: empresa[0].empresaId, empresaId}, entidade: empresa[0].empresaId}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:patchEmpresasBloquear").error(`Erro ao bloquear o empresa ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.patchEmpresasDesbloquear = async function(empresaId, req) {
  try {

      const actualizado_em = new Date()
      .toISOString()
      .replace('T',' ')
      .substr(0,19)

      logger("SERVIDOR:patchEmpresasDesbloquear").info(`Verificar o empresa pelo Id ${empresaId}`)
      const empresa = await database('empresa').where({empresaId})

      if(!empresa.length){
        logger("SERVIDOR:patchEmpresasDesbloquear").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:patchEmpresasDesbloquear").info(`Actualizando para desbloquear`)
      await database('empresa').where({empresaId}).update({bloqueioEmpresa:'1', actualizado_em})
      
      logger("SERVIDOR:patchEmpresasDesbloquear").info(`Empresa desbloqueado com sucesso`)
      const rs = response("sucesso", 202, "Empresa desbloqueado com sucesso", "json", {
        notification: {efeito: {empresa: empresa[0].nomeEmpresa, emailEmpresa: empresa[0].emailEmpresa}, para:"desbloququeioDeContaADM", mensagem: 'null', canal:"emailEmpresa"},
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "ACTIVE" , tabela: "CLIENTES", informacao: {entidade: empresa[0].empresaId, empresaId}, entidade: empresa[0].empresaId}
      });
      return rs
    
  } catch (erro) {
    console.log(erro)
    logger("SERVIDOR:patchEmpresasDesbloquear").error(`Erro ao desbloquear o empresa ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
  }
    
}

module.exports.configurarReporEmpresas = async function(empresaId,  tentativas_login , req) {
  try {

      logger("SERVIDOR:configurarReporEmpresas").debug(`Verificar a existencia do empresa`)
      const empresa = await database('empresa').where({empresaId})

      if(!empresa.length){
        logger("SERVIDOR:configurarReporEmpresas").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      logger("SERVIDOR:configurarReporEmpresas").debug(`Actualizar os numeros de tentativas de login no padrão`)
      await database('configuracoesEmpresa').where({empresaIdConf: empresa[0].empresaId}).update({tentativas_login: process.env.LIMITE_TENTATIVAS_LOGIN})

      logger("SERVIDOR:configurarReporEmpresas").debug(`Actualizar o empresa para o estado de novo empresa`)
      await database('empresa').where({empresaId}).update({nova_empresa:"1"})

      logger("SERVIDOR:configurarReporEmpresas").info(`A conta da entidade foi reposta com sucesso`)
      const rs = response("sucesso", 202, "A conta da entidade foi reposta com sucesso", "json", {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "PATH", tabela: "CONFIGURACOES", informacao: {tentativas_login, entidade: empresa[0].empresaId, empresaId}, entidade: empresa[0].empresaId}
      });
      return rs
    
  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:configurarReporEmpresas").error(`Erro ao repor o empresa ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}

module.exports.deleteEmpresas = async function(empresaId, req) { 
  try {

      logger("SERVIDOR:deleteEmpresas").debug(`Verificar se o empresa é do serviço GPO`)
      const clienteVerify = await database('empresa').where({empresaId}).orWhere({gpo_numero_comerciante:empresaId})
      let entidade = ""

      if(!clienteVerify.length){
        logger("SERVIDOR:deleteEmpresas").info("Empresa não foi encontrado")
        const rs = response("erro", 409, "Empresa não foi encontrado");
        return rs    
      }

      if(clienteVerify.length){
        empresaId = clienteVerify[0].empresaId
        entidade = clienteVerify[0].empresaId
      }   

      logger("SERVIDOR:deleteEmpresas").debug(`Verificar a existência do empresa`)
      const empresa = await database('empresa').where({empresaId})
      
      logger("SERVIDOR:deleteEmpresas").debug(`Verificar se o empresa tem referências geradas e usadas`)
      const clientesReferences = await database('empresaIds').where({entidade_cliente: empresa[0].empresaId});
      
      if(clientesReferences.length){
        logger("SERVIDOR:deleteEmpresas").info(`Empresa não exluido. Tem empresaIds geradas`)
        const rs = response("erro", 409, "Empresa não exluido. Tem empresaIds geradas");
        return rs
      }

      logger("SERVIDOR:deleteEmpresas").debug(`Verificar se o empresa tem historico de pagamentos`)
      const clientesPagamentos = await database('pagamentos').where({empresaId: empresa[0].empresaId});

      if(clientesPagamentos.length){
        logger("SERVIDOR:deleteEmpresas").info(`Empresa não exluido. Tem pagamentos feitos`)
        const rs = response("erro", 409, "Empresa não exluido. Tem pagamentos feitos");
        return rs
      }

      logger("SERVIDOR:deleteEmpresas").debug(`Á apagar o empresa`)
      await database('empresa').where({empresaId}).del() 

      logger("SERVIDOR:deleteEmpresas").info("Empresa exluido com sucesso")
      const rs = response("sucesso", 202, "Empresa exluido com sucesso", "json", {
        gpo_comerciante_hash: empresa[0].gpo_comerciante_hash,      
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DELETE" , tabela: "CLIENTES", informacao: {entidade: empresa[0].empresaId, empresaId}, entidade: empresa[0].empresaId}
      });
      return rs

  } catch (erro) {
      console.log(erro)
      logger("SERVIDOR:Empresas").error(`Erro ao deletar o empresa  ${erro.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
    
}